/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#include "XMLTypes.h"
#include <stdlib.h>

#define MAX_STRING_SIZE 21
#define BASE_10 10

using namespace CimXMLTypesNamespace;

// utility functions
static string ToBase64(unsigned char *in, int inlen)
{
 if(!in)
    {
      return "";
    }
  char* out = new char[((inlen + 1) * 4) / 3 + 3];
  char* out_start = out;
  // Base 64 Related
  const char base64digits[] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
  int   iTotalLength = 0;
  for (; inlen >= 3; inlen -= 3)
    {
      iTotalLength += 4;
      *out++ = base64digits[in[0] >> 2];
      *out++ = base64digits[((in[0] << 4) & 0x30) | (in[1] >> 4)];
      *out++ = base64digits[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
      *out++ = base64digits[in[2] & 0x3f];
      in += 3;
    }

  if (inlen > 0)
    {
      unsigned char fragment;

      iTotalLength += 4;

      *out++ = base64digits[in[0] >> 2];
      fragment = (in[0] << 4) & 0x30;
      if (inlen > 1)
	fragment |= in[1] >> 4;

      *out++ = base64digits[fragment];
      *out++ = (inlen < 2) ? '=' : base64digits[(in[1] << 2) & 0x3c];
      *out++ = '=';
    }

  *out = '\0';
  string tmp = out_start;
  delete[] out_start;
  return tmp;
}

static unsigned char A4toB3(unsigned char *a,
							unsigned char *b)
{
  char bChars[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
  char aChars[256];
  memset(aChars, 0xff, 256);
  for(int i=0; i<(int)sizeof(bChars); i++)
    {
      aChars[(int)bChars[i]] = (char)i;
    }
  unsigned long l = (a[0] << 18) | (a[1] << 12) | (a[2] << 6) | (a[3]);
  b[0] = (char)((l >> 16)& 0xFF);
  b[1] = (char)((l >> 8 )& 0xFF);
  b[2] = (char)((l      )& 0xFF);
  if (a[2] == 64) // '='
    return 1;
  else
    if (a[3] == 64) // '='
      return 2;
  return 3;
}

static void FromBase64(const unsigned char * inStr, unsigned char * outStr,
					   unsigned int  * outStrSize)
{
  if(!inStr || !outStr) return;
  unsigned int i, j;
  unsigned int size = 0;
  unsigned char a[4], *p=outStr, c;

  char bChars[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
  char aChars[256];
  memset(aChars, 0xff, 256);
  for(i=0; i<sizeof(bChars); i++)
    {
      aChars[(int)bChars[i]] = i;
    }

  for(i=0, j=0; inStr[i] != 0; i++)
    {
      c = aChars[inStr[i]];
      if (0xff != c)
	{
	  a[j & 3] = c;
	  j++;
	  if ((j & 3) == 0)
	    {
	      size += A4toB3(a, p);
	      p+=3;
	    }
	}
    }

  // final
  if ((j & 3) != 0)
    {
      size += A4toB3(a, p);
    }

  if(outStrSize)
    *outStrSize = size;
}

Base64::Base64() {
  data = NULL;
  len = 0;
}

Base64::Base64(const Base64& b)
{
  data = NULL;
  len = 0;
  SetData(b.data, b.len);
}

Base64::Base64(unsigned char *buffer, unsigned int blen)
{
  data = NULL;
  len = 0;
  SetData(buffer, blen);
}

Base64::Base64(const string& input)
{
  data = NULL;
  len = 0;
  SetData(input);
}

Base64::~Base64()
{
  if(data)
    {
      delete[] data;
      data = NULL;
    }
}

string Base64::ToString() const
{
  return ToBase64(data, len);
}

void Base64::SetData(const string& input)
{
  if(data)
    {
      delete[] data;
      data = NULL;
    }
  data = new unsigned char[((input.length() + (4 - (input.length() % 4))) * 3/4 + 1)];
  FromBase64((const unsigned char*)input.c_str(), data, &len);
}

void Base64::SetData(unsigned char *buffer, unsigned int blen)
{
  if(buffer)
    {
      if(data)
	{
	  delete[] data;
	  data = NULL;
	  }
      len = blen;
      data = new unsigned char[len];
      memcpy(data, buffer, len);
    }
}

Base64& Base64::operator =(const Base64& other)
{
  if (this == &other) {
    return *this;
  }
  if (data && len > 0)
    {
      delete[] data;
      data = NULL;
    }
  len = other.len;
    if (len > 0) {
      data = new unsigned char[len];
      memcpy(data, other.data, len);
    }
    return *this;
}

void XMLTypeConverter::StringToType(const string& str, int& t)
{
	if(!StringToInteger(str, t))
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type int.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const int& str)
{
	return IntegerToString(str);
}

void XMLTypeConverter::StringToType(const string& str, unsigned int& t)
{
	if(!StringToUInteger(str, t))
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type unsigned int.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const unsigned int& str)
{
	return UIntegerToString(str);
}

void XMLTypeConverter::StringToType(const string& str, char& t)
{
	if(!StringToInteger(str, t))
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type char.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const char& str)
{
	return IntegerToString(str);
}

void XMLTypeConverter::StringToType(const string& str, unsigned char& t)
{
	if(!StringToUInteger(str, t))
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type unsigned char.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const unsigned char& str)
{
	return UIntegerToString(str);
}

void XMLTypeConverter::StringToType(const string& str, short& t)
{
	if(!StringToInteger(str, t))
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type short.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const short& str)
{
	return IntegerToString(str);
}

void XMLTypeConverter::StringToType(const string& str, unsigned short& t)
{
	if(!StringToUInteger(str, t))
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type unsigned short.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const unsigned short& str)
{
	return UIntegerToString(str);
}

void XMLTypeConverter::StringToType(const string& str, Int64& t)
{
#ifdef _WIN32
  t= _atoi64(str.c_str());
#else
  t= strtoll(str.c_str(),NULL,BASE_10);
#endif
}

string XMLTypeConverter::TypeToString(const Int64& str)
{
  char buffer[MAX_STRING_SIZE];
#ifdef _WIN32
  _i64toa_s(str, buffer, MAX_STRING_SIZE, BASE_10);
#else
  sprintf(buffer, "%lld",str);
#endif
  return string(buffer);
}

void XMLTypeConverter::StringToType(const string& str, Uint64& t)
{
#ifdef _WIN32
  t = _strtoui64(str.c_str(), 0,0);
#else
  sscanf(str.c_str(),"%llu", &t);
#endif
}

string XMLTypeConverter::TypeToString(const Uint64& str)
{
  char buffer[MAX_STRING_SIZE];
#ifdef _WIN32
  _ui64toa_s(str, buffer, MAX_STRING_SIZE, BASE_10);
#else
  sprintf(buffer, "%llu",str);
#endif
  return buffer;
}

void XMLTypeConverter::StringToType(const string& str, bool& t)
{
	if(!str.compare("true") || !str.compare("1"))
	{
		t = true;
	}
	else if(!str.compare("false") || !str.compare("0"))
	{
		t = false;
	}
	else
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type boolean.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const bool& str)
{
	return (str ? "true" : "false");
}

void XMLTypeConverter::StringToType(const string& str, Base64& t)
{
	t.SetData(str);
}

string XMLTypeConverter::TypeToString(const Base64& buf)
{
	return buf.ToString();
}

void XMLTypeConverter::StringToType(const string& str, double& t)
{
	// check the string
	unsigned int i = (str.length() > 0 && str.at(0) == CHAR_MINUS) ? 1 : 0;
	bool pointFound = false;
	bool eFound = false;
	bool allZeros = true;
	char c;
	bool stat = true;
	for(; i < str.length(); i++)
	{
		c = str.at(i);
		if(c == '.')
		{
			if(pointFound)
			{
				stat = false;
				break;
			}
			pointFound = true;
			continue;
		}
		if(c == 'e' || c == 'E')
		{
			eFound = true;
			break;
		}
		if (c < CHAR_0 || c > CHAR_9)
		{
			stat = false;
		}
		if(c != CHAR_0)
		{
			allZeros = false;
		}
	}

	if(stat && eFound)
	{
		i++;
		c = str.at(i);
		if(c == '-' || c== '+')
		{
			i++;
		}
		for(; i < str.length(); i++)
		{
			c = str.at(i);
			if (c < CHAR_0 || c > CHAR_9)
			{
				stat = false;
			}
		}
	}

	if(stat && allZeros)
	{
		t = 0.0;
		return;
	}

	if(stat)
	{
		t = atof(str.c_str());
		if(t == 0.0)
		{
			stat = false;
		}
	}
	if(!stat)
	{
		throw XMLTypeConvertionException(string("The string ").append(str).append(" could not be converted to type double.").c_str());
	}
}

string XMLTypeConverter::TypeToString(const double& str)
{
	if(str == 0)
	{
		return "0";
	}
	return ToString(str);
}

void XMLTypeConverter::StringToType(const string& str, string& t)
{
	t = str;
}

string XMLTypeConverter::TypeToString(const string& str)
{
	return str;
}
