/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <Unity/Application/application_manager.h>

#include <Unity/Application/applicationcontroller.h>
#include <Unity/Application/taskcontroller.h>
#include <Unity/Application/proc_info.h>

#include <core/posix/linux/proc/process/oom_score_adj.h>

#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include <QSignalSpy>

#include "mock_application_controller.h"
#include "mock_desktop_file_reader.h"
#include "mock_oom_controller.h"
#include "mock_process_controller.h"
#include "mock_proc_info.h"
#include "mock_session.h"
#include "mock_focus_controller.h"

using namespace unitymir;

class ApplicationManagerTests : public ::testing::Test
{
public:
    ApplicationManagerTests()
        : processController{
            QSharedPointer<ProcessController::OomController> (
                &oomController,
                [](ProcessController::OomController*){})
        },
        applicationManager{
            QSharedPointer<TaskController>{
                new TaskController(
                    nullptr,
                    QSharedPointer<ApplicationController>(
                        &appController,
                        [](ApplicationController*){}),
                    QSharedPointer<ProcessController>(
                        &processController,
                        [](ProcessController*){})
                    )},
            QSharedPointer<DesktopFileReader::Factory>(
                &desktopFileReaderFactory,
                [](DesktopFileReader::Factory*){}),
            QSharedPointer<ProcInfo>(&procInfo,[](ProcInfo *){}),
            std::shared_ptr<mir::shell::FocusController>(&focusController, [](void*){}),
            QSize(400,400)
        }
    {
    }
    testing::NiceMock<testing::MockOomController> oomController;
    testing::NiceMock<testing::MockProcessController> processController;
    testing::NiceMock<testing::MockApplicationController> appController;
    testing::NiceMock<testing::MockProcInfo> procInfo;
    testing::NiceMock<testing::MockDesktopFileReaderFactory> desktopFileReaderFactory;
    testing::NiceMock<testing::MockFocusController> focusController;
    ApplicationManager applicationManager;
};

TEST_F(ApplicationManagerTests, SuspendingAndResumingARunningApplicationResultsInOomScoreAdjustment)
{
    using namespace ::testing;

    const QString appId("com.canonical.does.not.exist");

    EXPECT_CALL(appController, startApplicationWithAppIdAndArgs(_, _)).Times(1);

    EXPECT_CALL(desktopFileReaderFactory, createInstanceForAppId(appId)).Times(1);

    EXPECT_CALL(processController, sigStopProcessGroupForPid(_)).Times(1);
    EXPECT_CALL(processController, sigContinueProcessGroupForPid(_)).Times(1);
    EXPECT_CALL(oomController, ensureProcessUnlikelyToBeKilled(_)).Times(1);
    EXPECT_CALL(oomController, ensureProcessLikelyToBeKilled(_)).Times(1);

    auto application = applicationManager.startApplication(
                appId,
                ApplicationManager::NoFlag,
                QStringList());

    application->suspend();
    application->resume();
}

// Currently disabled as we need to make sure that we have a corresponding mir session, too.
TEST_F(ApplicationManagerTests, DISABLED_FocusingRunningApplicationResultsInOomScoreAdjustment)
{
    using namespace ::testing;

    const QString appId("com.canonical.does.not.exist");

    QSet<QString> appIds;

    for (unsigned int i = 0; i < 50; i++)
    {
        QString appIdFormat("%1.does.not.exist");
        auto appId = appIdFormat.arg(i);

        auto application = applicationManager.startApplication(
                    appId,
                    ApplicationManager::NoFlag,
                    QStringList());

        std::shared_ptr<mir::shell::Session> mirSession = std::make_shared<MockSession>(appIdFormat.toStdString(), i);
        applicationManager.onSessionStarting( mirSession );

        EXPECT_NE(nullptr, application);

        appIds.insert(appId);
        auto it = appController.children.find(appId);
        if (it != appController.children.end())
            EXPECT_CALL(oomController,
                        ensureProcessUnlikelyToBeKilled(it->pid())).Times(1);
    }

    for (auto appId : appIds)
    {
        applicationManager.focusApplication(appId);
    }
}


TEST_F(ApplicationManagerTests,bug_case_1240400_second_dialer_app_fails_to_authorize_and_gets_mixed_up_with_first_one)
{
    using namespace ::testing;
    std::shared_ptr<mir::shell::Surface> aSurface(nullptr);
    quint64 firstProcId = 5921;
    quint64 secondProcId = 5922;
    const char dialer_app_id[] = "dialer-app";
    QByteArray cmdLine( "/usr/bin/dialer-app --desktop_file_hint=dialer-app");
    QByteArray secondcmdLine( "/usr/bin/dialer-app");

    EXPECT_CALL(procInfo,command_line(firstProcId))
        .Times(1)
        .WillOnce(Return(cmdLine));
    EXPECT_CALL(procInfo,command_line(secondProcId))
        .Times(1)
        .WillOnce(Return(secondcmdLine));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> mirSession = std::make_shared<MockSession>(dialer_app_id, firstProcId);
    applicationManager.authorizeSession(firstProcId, authed);
    EXPECT_EQ(true, authed);
    applicationManager.onSessionStarting(mirSession);
    applicationManager.onSessionCreatedSurface(mirSession.get(),aSurface);
    Application * app = applicationManager.findApplication(dialer_app_id);
    EXPECT_NE(nullptr,app);

    // now a second session without desktop file is launched:
    applicationManager.authorizeSession(secondProcId, authed);
    applicationManager.onProcessStartReportReceived(dialer_app_id, true);

    EXPECT_EQ(false,authed);
    EXPECT_EQ(app,applicationManager.findApplication(dialer_app_id));
    EXPECT_EQ(QString(dialer_app_id),applicationManager.focusedApplicationId());
}

TEST_F(ApplicationManagerTests,application_dies_while_starting)
{
    using namespace ::testing;
    quint64 procId = 5921;
    const char app_id[] = "my-app";
    QByteArray cmdLine( "/usr/bin/my-app --desktop_file_hint=my-app");

    EXPECT_CALL(procInfo,command_line(procId))
        .Times(1)
        .WillOnce(Return(cmdLine));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> mirSession = std::make_shared<MockSession>(app_id, procId);
    applicationManager.authorizeSession(procId, authed);
    applicationManager.onSessionStarting(mirSession);
    Application * beforeFailure = applicationManager.findApplication(app_id);
    applicationManager.onProcessStartReportReceived(app_id,true);
    Application * afterFailure = applicationManager.findApplication(app_id);

    EXPECT_EQ(true, authed);
    EXPECT_NE(nullptr, beforeFailure);
    EXPECT_EQ(nullptr, afterFailure);
}

TEST_F(ApplicationManagerTests,application_start_failure_after_starting)
{
    using namespace ::testing;
    quint64 procId = 5921;
    std::shared_ptr<mir::shell::Surface> aSurface(nullptr);
    const char app_id[] = "my-app";
    QByteArray cmdLine( "/usr/bin/my-app --desktop_file_hint=my-app");

    EXPECT_CALL(procInfo,command_line(procId))
        .Times(1)
        .WillOnce(Return(cmdLine));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> mirSession = std::make_shared<MockSession>(app_id, procId);
    applicationManager.authorizeSession(procId, authed);
    applicationManager.onSessionStarting(mirSession);
    Application * beforeFailure = applicationManager.findApplication(app_id);
    applicationManager.onSessionCreatedSurface(mirSession.get(), aSurface);
    applicationManager.onProcessStartReportReceived(app_id, true);
    Application * afterFailure = applicationManager.findApplication(app_id);

    EXPECT_EQ(true, authed);
    EXPECT_NE(nullptr, beforeFailure);
    EXPECT_EQ(beforeFailure, afterFailure);
}

TEST_F(ApplicationManagerTests,bug_case_1281075_session_ptrs_always_distributed_to_last_started_app)
{
    using namespace ::testing;
    quint64 first_procId = 5921;
    quint64 second_procId = 5922;
    quint64 third_procId = 5923;
    std::shared_ptr<mir::shell::Surface> aSurface(nullptr);
    const char first_app_id[] = "app1";
    QByteArray first_cmdLine( "/usr/bin/app1 --desktop_file_hint=app1");
    const char second_app_id[] = "app2";
    QByteArray second_cmdLine( "/usr/bin/app2--desktop_file_hint=app2");
    const char third_app_id[] = "app3";
    QByteArray third_cmdLine( "/usr/bin/app3 --desktop_file_hint=app3");

    EXPECT_CALL(procInfo,command_line(first_procId))
        .Times(1)
        .WillOnce(Return(first_cmdLine));

    ON_CALL(appController,appIdHasProcessId(_,_)).WillByDefault(Return(false));

    EXPECT_CALL(procInfo,command_line(second_procId))
        .Times(1)
        .WillOnce(Return(second_cmdLine));

    EXPECT_CALL(procInfo,command_line(third_procId))
        .Times(1)
        .WillOnce(Return(third_cmdLine));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> first_session = std::make_shared<MockSession>("Oo", first_procId);
    std::shared_ptr<mir::shell::Session> second_session = std::make_shared<MockSession>("oO", second_procId);
    std::shared_ptr<mir::shell::Session> third_session = std::make_shared<MockSession>("OO", third_procId);
    applicationManager.authorizeSession(first_procId, authed);
    applicationManager.authorizeSession(second_procId, authed);
    applicationManager.authorizeSession(third_procId, authed);
    applicationManager.onSessionStarting(first_session);
    applicationManager.onSessionStarting(third_session);
    applicationManager.onSessionStarting(second_session);

    Application * firstApp = applicationManager.findApplication(first_app_id);
    Application * secondApp = applicationManager.findApplication(second_app_id);
    Application * thirdApp = applicationManager.findApplication(third_app_id);

    EXPECT_EQ(first_session, firstApp->session());
    EXPECT_EQ(second_session, secondApp->session());
    EXPECT_EQ(third_session, thirdApp->session());
}

TEST_F(ApplicationManagerTests,two_session_on_one_application)
{
    using namespace ::testing;
    quint64 a_procId = 5921;
    const char an_app_id[] = "some_app";
    QByteArray a_cmd( "/usr/bin/app1 --desktop_file_hint=some_app");

    ON_CALL(procInfo,command_line(_)).WillByDefault(Return(a_cmd));

    ON_CALL(appController,appIdHasProcessId(_,_)).WillByDefault(Return(false));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> first_session = std::make_shared<MockSession>("Oo", a_procId);
    std::shared_ptr<mir::shell::Session> second_session = std::make_shared<MockSession>("oO", a_procId);
    applicationManager.authorizeSession(a_procId, authed);

    applicationManager.onSessionStarting(first_session);
    applicationManager.onSessionStarting(second_session);

    Application * the_app = applicationManager.findApplication(an_app_id);

    EXPECT_EQ(true, authed);
    EXPECT_EQ(second_session, the_app->session());
}

TEST_F(ApplicationManagerTests,upstart_launching_sidestage_app_on_phone_forced_into_mainstage)
{
    using namespace ::testing;
    QString appId("sideStage");

    auto mockDesktopFileReader = new NiceMock<MockDesktopFileReader>(appId);
    ON_CALL(*mockDesktopFileReader, loaded()).WillByDefault(Return(true));
    ON_CALL(*mockDesktopFileReader, stageHint()).WillByDefault(Return("SideStage"));

    ON_CALL(desktopFileReaderFactory, createInstanceForAppId(appId)).WillByDefault(Return(mockDesktopFileReader));

    // mock upstart launching an app which reports itself as sidestage, but we're on phone
    applicationManager.onProcessStartReportReceived(appId, false);

    // ensure the app stage is overridden to be main stage
    Application* theApp = applicationManager.findApplication(appId);
    ASSERT_NE(theApp, nullptr);
    EXPECT_EQ(Application::MainStage, theApp->stage());
}

TEST_F(ApplicationManagerTests,two_session_on_one_application_after_starting)
{
    using namespace ::testing;
    quint64 a_procId = 5921;
    const char an_app_id[] = "some_app";
    QByteArray a_cmd( "/usr/bin/app1 --desktop_file_hint=some_app");
    std::shared_ptr<mir::shell::Surface> aSurface(nullptr);

    ON_CALL(procInfo,command_line(_)).WillByDefault(Return(a_cmd));

    ON_CALL(appController,appIdHasProcessId(_,_)).WillByDefault(Return(false));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> first_session = std::make_shared<MockSession>("Oo", a_procId);
    std::shared_ptr<mir::shell::Session> second_session = std::make_shared<MockSession>("oO", a_procId);
    applicationManager.authorizeSession(a_procId, authed);

    applicationManager.onSessionStarting(first_session);
    applicationManager.onSessionCreatedSurface(first_session.get(), aSurface);
    applicationManager.onSessionStarting(second_session);

    Application * the_app = applicationManager.findApplication(an_app_id);

    EXPECT_EQ(true, authed);
    EXPECT_EQ(Application::Running, the_app->state());
    EXPECT_EQ(first_session, the_app->session());
}

TEST_F(ApplicationManagerTests,suspended_suspends_focused_app)
{
    using namespace ::testing;
    quint64 a_procId = 5921;
    const char an_app_id[] = "some_app";
    QByteArray a_cmd( "/usr/bin/app1 --desktop_file_hint=some_app");
    std::shared_ptr<mir::shell::Surface> aSurface(nullptr);

    ON_CALL(procInfo,command_line(_)).WillByDefault(Return(a_cmd));

    ON_CALL(appController,appIdHasProcessId(_,_)).WillByDefault(Return(false));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> first_session = std::make_shared<MockSession>("Oo", a_procId);
    std::shared_ptr<mir::shell::Session> second_session = std::make_shared<MockSession>("oO", a_procId);
    applicationManager.authorizeSession(a_procId, authed);

    applicationManager.onSessionStarting(first_session);
    applicationManager.onSessionCreatedSurface(first_session.get(), aSurface);
    applicationManager.onSessionStarting(second_session);

    Application * the_app = applicationManager.findApplication(an_app_id);

    EXPECT_EQ(Application::Running, the_app->state());

    applicationManager.setSuspended(true);

    EXPECT_EQ(Application::Suspended, the_app->state());

    applicationManager.setSuspended(false);

    EXPECT_EQ(Application::Running, the_app->state());

}

TEST_F(ApplicationManagerTests,requestFocusApplication)
{
    using namespace ::testing;
    quint64 first_procId = 5921;
    quint64 second_procId = 5922;
    quint64 third_procId = 5923;
    std::shared_ptr<mir::shell::Surface> aSurface(nullptr);
    QByteArray first_cmdLine( "/usr/bin/app1 --desktop_file_hint=app1");
    QByteArray second_cmdLine( "/usr/bin/app2--desktop_file_hint=app2");
    QByteArray third_cmdLine( "/usr/bin/app3 --desktop_file_hint=app3");

    EXPECT_CALL(procInfo,command_line(first_procId))
        .Times(1)
        .WillOnce(Return(first_cmdLine));

    ON_CALL(appController,appIdHasProcessId(_,_)).WillByDefault(Return(false));

    EXPECT_CALL(procInfo,command_line(second_procId))
        .Times(1)
        .WillOnce(Return(second_cmdLine));

    EXPECT_CALL(procInfo,command_line(third_procId))
        .Times(1)
        .WillOnce(Return(third_cmdLine));

    bool authed = true;

    std::shared_ptr<mir::shell::Session> first_session = std::make_shared<MockSession>("Oo", first_procId);
    std::shared_ptr<mir::shell::Session> second_session = std::make_shared<MockSession>("oO", second_procId);
    std::shared_ptr<mir::shell::Session> third_session = std::make_shared<MockSession>("OO", third_procId);
    applicationManager.authorizeSession(first_procId, authed);
    applicationManager.authorizeSession(second_procId, authed);
    applicationManager.authorizeSession(third_procId, authed);
    applicationManager.onSessionStarting(first_session);
    applicationManager.onSessionStarting(third_session);
    applicationManager.onSessionStarting(second_session);

    QSignalSpy spy(&applicationManager, SIGNAL(focusRequested(const QString &)));

    applicationManager.requestFocusApplication("app3");

    EXPECT_EQ(spy.count(), 1);

    QList<QVariant> arguments = spy.takeFirst(); // take the first signal
    EXPECT_EQ(arguments.at(0).toString(), "app3");
}
