/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "themebutton.h"
#include <QVBoxLayout>
#include <QLabel>
#include <QDebug>
#include <QPainter>

ThemeButton::ThemeButton(const QString &name, QWidget *parent):
    QPushButton(parent),
    themename(name)
{
    mIconLabel = new ThemeIconLabel(this);
    initUi();
    setConnect();
}

ThemeButton::ThemeButton(const QString &name, const QPixmap &pixmap, QWidget *parent):
    QPushButton(parent),
    themename(name)
{
    mIconLabel = new ThemeIconLabel(pixmap, this);
    initUi();
    setConnect();
}


ThemeButton::ThemeButton(QString name, QStringList iStringList, QWidget *parent):
    themename(name),
    QPushButton(parent)
{
    mIconLabel = new ThemeIconLabel( iStringList, this);
    initUi();
    setConnect();
}

ThemeButton::ThemeButton(QString name, QList<QPixmap> iconList, QWidget *parent):
    themename(name),
    QPushButton(parent)
{
    mIconLabel = new ThemeIconLabel( iconList, this);
    initUi();
    setConnect();
}

void ThemeButton::initUi()
{
    clickedBtnStyle = getStyle(2, mRadius);
    enterBtnStyle = getStyle(1, mRadius);
    leaveBtnStyle = getStyle(0, mRadius);

    setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
    setCheckable(true);
    setFixedWidth(mWidth);
    QVBoxLayout *baseVerLayout = new QVBoxLayout(this);
    baseVerLayout->setSpacing(8);
    baseVerLayout->setMargin(0);

    mIconLabel->setFixedSize(QSize(mWidth, mHeigh - 32));
    mIconLabel->setAttribute(Qt::WA_DeleteOnClose);
    mIconLabel->setStyleSheet(leaveBtnStyle);

    QHBoxLayout *bottomHorLayout = new QHBoxLayout;
    bottomHorLayout->setSpacing(8);
    bottomHorLayout->setMargin(0);

    mNameLabel = new QLabel(this);
    mNameLabel->setText(themename);
    mNameLabel->setAlignment(Qt::AlignTop);

    bottomHorLayout->addStretch();
    bottomHorLayout->setContentsMargins(0, 0, 0, 0);
    bottomHorLayout->addWidget(mNameLabel);
    bottomHorLayout->addStretch();

    baseVerLayout->addWidget(mIconLabel);
    baseVerLayout->addLayout(bottomHorLayout);

    this->setLayout(baseVerLayout);
}

void ThemeButton::setConnect()
{
    connect(this, &ThemeButton::clicked, this, [=]() {
        mIconLabel->setStyleSheet(clickedBtnStyle);
    });

    connect(this, &ThemeButton::btnCheckedSignals, [=](bool status) {
        this->setChecked(status);
        if (!status)
            mIconLabel->setStyleSheet(leaveBtnStyle);
    });

    connect(mIconLabel,&ThemeIconLabel::enterWidget,[=](){
        if (!this->isChecked()) {
            mIconLabel->setStyleSheet(enterBtnStyle);
        }
    });

    connect(mIconLabel,&ThemeIconLabel::leaveWidget,[=](){
        if (!this->isChecked()) {
            mIconLabel->setStyleSheet(leaveBtnStyle);
        }
    });
}

void ThemeButton::setRadius(int radius)
{
    mRadius = radius;
    mIconLabel->setRadius(radius);

    clickedBtnStyle = getStyle(2, mRadius);
    enterBtnStyle = getStyle(1, mRadius);
    leaveBtnStyle = getStyle(0, mRadius);
}

void ThemeButton::setRect(QSize size)
{
    mWidth = size.width();
    mHeigh = size.height();
    setFixedWidth(mWidth);
    mIconLabel->setFixedSize(QSize(mWidth, mHeigh - 32));

}

void ThemeButton::setPixmap(const QPixmap &pixmap)
{    
    mIconLabel->setIcon(pixmap);
}

void ThemeButton::setBtnClicked(bool status)
{
    this->setChecked(status);
    if (!this->isChecked()) {
        mIconLabel->setStyleSheet(leaveBtnStyle);
    } else {
        mIconLabel->setStyleSheet(clickedBtnStyle);
    }
}

QString ThemeButton::getStyle(int border, int radius)
{
    return QString("ThemeIconLabel{background: palette(window); border-color: palette(highlight);"
                   "border-width: %1px;border-style: solid;border-radius: %2px}").arg(border).arg(radius);
}

void ThemeButton::paintEvent(QPaintEvent *event)
{
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);  // 反锯齿;
    painter.setClipping(true);
    painter.setPen(Qt::transparent);

    painter.drawRoundedRect(this->rect(), mRadius, mRadius);


    if (height() != mIconLabel->height() + 8 + mNameLabel->height())
        this->setFixedHeight(mIconLabel->height() + 8 + mNameLabel->height());

}
