///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file PluginClassDescriptor.h
 * \brief Contains the definition of the Core::PluginClassDescriptor class.
 */

#ifndef __OVITO_PLUGINCLASS_DESCRIPTOR_H
#define __OVITO_PLUGINCLASS_DESCRIPTOR_H

#include <core/Core.h>

namespace Core {

class PluginClassDescriptor;		// defined below
class PluginClass;					// defined below
class PropertyFieldDescriptor;		// defined in PropertyFieldDescriptor.h
class ObjectSaveStream;			// defined in ObjectSaveStream.h
class ObjectLoadStream;			// defined in ObjectLoadStream.h
class Plugin;						// defined in Plugin.h

/// The identifier of the built-in core plugin.
/// \sa Plugin::isCore(), Plugin::pluginId()
#define CORE_PLUGIN_ID		"Core"

/**
 * \brief Describes a class defined by a plugin.
 *
 * Each class provided by a plug-in is decribed by such a descriptor structure
 * that is loaded from the plug-in's manifest file.
 *
 * \author Alexander Stukowski
 * \sa Plugin, PluginClass, PluginClassInfo
 */
class CORE_DLLEXPORT PluginClassDescriptor
{
public:
	/// \brief Returns the name of the class.
	/// \return The name of the class (without namespace qualifier). This name is only unique
	///         within one Plugin.
	const QString& name() const { return _name; }

	/// \brief Returns the plugin that contains the class.
	/// \return The plugin that defines this class.
	Plugin* plugin() const { return _plugin; }

	/// \brief Returns the descriptor of the base class this class is derived from.
	/// \return The descriptor of the base class or \c NULL if this is the root PluginClass.
	PluginClassDescriptor* baseClass() const { return _baseClass; }

	/// \brief Indicates whether this is an abstract class.
	/// \return \c true if this is an abstract class that cannot be instantiated using createInstance();
	///         \c false otherwise.
	/// \sa createInstance()
	bool isAbstract() const { return _isAbstract; }

	/// \brief Returns whether this class can be written to file.
	/// \return \c true if instances of this class can be serialized to a file;
	///         \c false otherwise.
	/// \note A class is only serializable if all its base classes are also serializable.
	bool isSerializable() const {
		OVITO_ASSERT_MSG(baseClass() == NULL || !(_isSerializable && !baseClass()->isSerializable()), "PluginClassDescriptor::IsSerializable", "Class derived from a non-serializable class has to be non-serializable too.");
		return _isSerializable;
	}

	/// \brief Returns whether this class is directly of indirectly derived from some other class.
	/// \return \c true if this class is a kind of (inherits from) the given class;
	///         \c false otherwise.
	/// \note This method returns \c true if the given class \a other is the class itself.
	bool isKindOf(const PluginClassDescriptor* other) const {
		for(const PluginClassDescriptor* c = this; c != NULL; c = c->baseClass())
			if(c == other) return true;
		return false;
	}

	/// \brief Creates an object of the appropriate kind.
	/// \param isLoading This specifies whether the object is being loaded from a file or just
	///                  created from scratch at runtime. This parameter is passed on to the
	///                  object constructor.
	/// \return The new instance of the class. The pointer can safely be cast to the appropriate C++ class type.
	/// \note The returned object pointer should be immediately wrapped in a \c intrusive_ptr<>
	///       after createInstance() returns.
	/// \throw Exception if the plugin failed to load.
	intrusive_ptr<PluginClass> createInstance(bool isLoading = false);

	/// \brief Returns the XML element from the manifest file that describes this class.
	/// \return The XML element. This can be used to access additional meta information about the class.
	/// \sa getMetaData()
	const QDomElement& classElement() const { return _classNode; }

	/// \brief Returns the XML element from the class' manifest that contains the
	///        metadata with the given name.
	/// \return The XML element with the name \a metadataName.
	/// \sa classElement()
	QDomElement getMetaData(const QString& metadataName) const { return _classNode.firstChildElement(metadataName); }

	/// \brief Returns the display name of this plugin class.
	/// \return The human readable name of this plugin class taken from the plugin manifest.
	QString schematicTitle() const;

	/// \brief Returns the first element of the linked list of reference fields defined for this class if it is a RefMaker derived class.
	const PropertyFieldDescriptor* firstPropertyField() const { return _firstPropertyField; }

	/// If this is the descriptor of a RefMaker-derived class then this method will return
	/// the reference field with the given identifier that has been defined in the RefMaker-derived
	/// class. If no such field is defined by that class then NULL is returned.
	/// Note that this method will NOT return reference fields that have been defined in
	/// super-classes.
	const PropertyFieldDescriptor* findPropertyField(const char* identifier) const;

	/// The descriptor of the root class of all plugin classes: PluginClass.
	static PluginClassDescriptor& rootClass;

	/// \brief Writes a class descriptor to the stream.
	/// \note This method is for internal use only.
	static void saveRTTI(ObjectSaveStream& stream, const PluginClassDescriptor* descriptor);

	/// \brief Loads a class descriptor from the stream.
	/// \throw Eexception if the class is not defined or the required plugin is not installed.
	/// \note This method is for internal use only.
	static PluginClassDescriptor* loadRTTI(ObjectLoadStream& stream);

protected:

	/// Constructor.
	PluginClassDescriptor(const QString& name, PluginClassDescriptor* baseClass, Plugin* plugin, const QDomElement& classNode, bool isAbstract, bool serializable);

	/// Destructor.
	virtual ~PluginClassDescriptor();

	/// \brief Creates an instance of the class described by this descriptor.
	/// \param isLoading This specifies whether the object is being loaded from a file or just
	///                  created from scratch at runtime.
	/// \return The new instance of the class. The pointer can safely be cast to the appropriate C++ class type.
	/// \throw Exception if the instance could not be created.
	virtual intrusive_ptr<PluginClass> createInstanceImpl(bool isLoading) = 0;

protected:

	/// The class name.
	QString _name;

	/// The plugin that contains the class.
	Plugin*	_plugin;

	/// The base class descriptor or NULL.
    PluginClassDescriptor* _baseClass;

	/// The next class descriptor that has the same parent as this one.
	PluginClassDescriptor* nextSibling;

	/// The first derived class.
	PluginClassDescriptor* firstChild;

	/// The next descriptor in the global linked list of all classes.
	PluginClassDescriptor* next;

	/// Indicates if this is an abstract class.
	bool _isAbstract;

	/// True if this class can be serialized.
	bool _isSerializable;

	/// The linked list of property fields if this is a RefMaker class.
	const PropertyFieldDescriptor* _firstPropertyField;

	/// The XML element from the manifest file that describes this class.
	QDomElement _classNode;

	friend class Plugin;
	friend class PluginManager;
};

};

#endif // __OVITO_PLUGINCLASS_DESCRIPTOR_H
