/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Stdlib includes


/////////////////////// Qt includes
#include <QString>


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/libXpertMassCore/globals.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{


// #include <libisospec++/isoSpec++.h>
//
// extern const int elem_table_atomicNo[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const double
// elem_table_probability[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const double elem_table_mass[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const int elem_table_massNo[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const int
// elem_table_extraNeutrons[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const char* elem_table_element[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const char* elem_table_symbol[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const bool elem_table_Radioactive[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];
// extern const double
// elem_table_log_probability[ISOSPEC_NUMBER_OF_ISOTOPIC_ENTRIES];


enum class Version1IsotopeFields
{
  ID             = 0,
  ELEMENT        = 1,
  SYMBOL         = 2,
  ATOMIC_NUMBER  = 3,
  MASS           = 4,
  MASS_NUMBER    = 5,
  EXTRA_NEUTRONS = 6,
  PROBABILITY    = 7,
  LN_PROBABILITY = 8,
  RADIOACTIVE    = 9,
  LAST           = 10,
};

enum class IsotopeFields
{
  NAME        = 0,
  SYMBOL      = 1,
  MASS        = 2,
  PROBABILITY = 3,
  LAST        = 4,
};

/*  BEGIN CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: Isotope
 */

class DECLSPEC Isotope: public QObject
{
  Q_OBJECT

  // We need the READ WRITE get/set functions because there is validation code.
  Q_PROPERTY(QString name READ getName WRITE setName)
  Q_PROPERTY(QString symbol READ getSymbol WRITE setSymbol)
  Q_PROPERTY(double mass READ getMass WRITE setMass)
  Q_PROPERTY(double probability READ getProbability WRITE setProbability)
  // No setter because it should be set by the validation function only.
  Q_PROPERTY(bool isValid READ isValid)

  public:
  enum class Version1IsotopeFields
  {
    ID             = 0,
    ELEMENT        = 1,
    SYMBOL         = 2,
    ATOMIC_NUMBER  = 3,
    MASS           = 4,
    MASS_NUMBER    = 5,
    EXTRA_NEUTRONS = 6,
    PROBABILITY    = 7,
    LN_PROBABILITY = 8,
    RADIOACTIVE    = 9,
    LAST           = 10,
  };
  Q_ENUM(Version1IsotopeFields)

  enum class IsotopeFields
  {
    NAME        = 0,
    SYMBOL      = 1,
    MASS        = 2,
    PROBABILITY = 3,
    LAST        = 4,
  };
  Q_ENUM(IsotopeFields)

  Q_INVOKABLE Isotope(QObject *parent = nullptr);
  Q_INVOKABLE Isotope(const QString &name,
                      const QString &symbol,
                      double mass,
                      double probability,
                      QObject *parent = nullptr);

  Q_INVOKABLE Isotope(const QString &text, QObject *parent = nullptr);

  // Pseudo copy constructor
  Isotope(const Isotope &other, QObject *parent = nullptr);
  Isotope(const Isotope *other_p, QObject *parent = nullptr);

  virtual ~Isotope();

  Q_INVOKABLE Isotope *clone(QObject *parent = nullptr) const;
  static Isotope *clone(const Isotope &other, QObject *parent = nullptr);

  Q_INVOKABLE bool initialize(const QString &name,
                              const QString &symbol,
                              double mass,
                              double probability);

  Q_INVOKABLE bool initialize(const QString &text);
  bool initializeVersion1(const QString &text);
  bool initializeVersion2(const QString &text);

  Q_INVOKABLE void clear();

  Q_INVOKABLE void setName(const QString &name);
  Q_INVOKABLE QString getName() const;

  Q_INVOKABLE void setSymbol(const QString &symbol);
  Q_INVOKABLE QString getSymbol() const;

  Q_INVOKABLE void setMass(double mass);
  Q_INVOKABLE double getMass() const;

  Q_INVOKABLE void setProbability(double probability);
  Q_INVOKABLE double getProbability() const;

  Q_INVOKABLE bool validate(ErrorList *error_list_p);
  bool isValid() const;

  bool operator=(const Isotope &other) const = delete;
  bool operator==(const Isotope &other) const;
  bool operator!=(const Isotope &other) const;

  Q_INVOKABLE QString toString() const;

  static void registerJsConstructor(QJSEngine *engine);

  protected:
  QString m_name       = "";
  QString m_symbol     = "";
  double m_mass        = -1.0;
  double m_probability = -1.0;
  bool m_isValid       = false;
};

typedef QSharedPointer<Isotope> IsotopeQSPtr;
typedef QSharedPointer<const Isotope> IsotopeCstQSPtr;

/*  END CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: Isotope
 */

} // namespace libXpertMassCore

MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassCore, Isotope)

} // namespace MsXpS

Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::Isotope);
extern int isotopeMetaTypeId;

Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::Isotope *);
extern int isotopePtrMetaTypeId;

Q_DECLARE_METATYPE(QSharedPointer<MsXpS::libXpertMassCore::Isotope>);
extern int isotopeSPtrMetaTypeId;

Q_DECLARE_METATYPE(QSharedPointer<const MsXpS::libXpertMassCore::Isotope>);
extern int isotopeCstSPtrMetaTypeId;
