/*
 *                            COPYRIGHT
 *
 *  pcb-rnd, interactive printed circuit board design
 *  Copyright (C) 2018 Tibor 'Igor2' Palinkas
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://www.repo.hu/projects/librnd
 *    lead developer: http://www.repo.hu/projects/librnd/contact.html
 *    mailing list: pcb-rnd (at) list.repo.hu (send "subscribe")
 */

#include "wt_preview.h"
#include "dialogs.h"
#include <librnd/hid/hid_attrib.h>

static gdl_list_t ltf_previews;

static int widget_depth(Widget w) {
	Arg args[1];
	int depth;

	XtSetArg(args[0], XtNdepth, &depth);
	XtGetValues(w, args, 1);
	return depth;
}

void rnd_ltf_preview_zoom_update(rnd_ltf_preview_t *pd)
{
	double z;
	Dimension w, h;

	pd->resized = 1;
	stdarg_n = 0;
	stdarg(XmNwidth, &w);
	stdarg(XmNheight, &h);
	XtGetValues(pd->pw, stdarg_args, stdarg_n);
	pd->v_width = w;
	pd->v_height = h;

	pd->zoom = (pd->x2 - pd->x1 + 1) / (double)pd->v_width;
	z = (pd->y2 - pd->y1 + 1) / (double)pd->v_height;
	if (pd->zoom < z)
		pd->zoom = z;

	pd->x = (pd->x1 + pd->x2) / 2 - pd->v_width * pd->zoom / 2;
	pd->y = (pd->y1 + pd->y2) / 2 - pd->v_height * pd->zoom / 2;

	/* might be called from within an user expose callback - should affect the rest of the drawing */
	if (pd->expose_lock) {
		xrnd.view_left_x = pd->x1;
		xrnd.view_top_y = pd->y1;
		xrnd.view_zoom = pd->zoom;
		xrnd.view_width = pd->x2;
		xrnd.view_height = pd->y2;
	}
}

void rnd_ltf_preview_getxy(rnd_ltf_preview_t *pd, int px, int py, rnd_coord_t *dst_x, rnd_coord_t *dst_y)
{
	rnd_coord_t x, y;

	if (rnd_conf.editor.view.flip_x)
		px = pd->v_width - px;
	x = rnd_round((double)px * pd->zoom + (double)pd->x1);

	if (rnd_conf.editor.view.flip_y)
		py = pd->v_height - py;
	y = rnd_round((double)py * pd->zoom + (double)pd->y1);

	*dst_x = x;
	*dst_y = y;
}


void rnd_ltf_preview_redraw(rnd_ltf_preview_t *pd)
{
	GC gc;
	XGCValues gcv;

	if (pd->expose_lock)
		return;

	memset(&gcv, 0, sizeof(gcv));
	gcv.graphics_exposures = 0;
	gc = XtGetGC(pd->pw, GCGraphicsExposures, &gcv);

	pd->expose_lock = 1;

	xrnd_render_preview(&lesstif_hid, XtDisplay(pd->pw), XtWindow(pd->pw),
		widget_depth(pd->pw), gc, pd->v_width, pd->v_height,
		pd->x1, pd->y1, pd->x2, pd->y2, pd->zoom, &pd->exp_ctx, 
		pd->flip_local, pd->flip_global, pd->flip_x, pd->flip_y);

	XtReleaseGC(pd->pw, gc);

	pd->expose_lock = 0;
}

void rnd_ltf_preview_callback(Widget da, rnd_ltf_preview_t *pd, XmDrawingAreaCallbackStruct *cbs)
{
	int reason = cbs != NULL ? cbs->reason : 0;
	lesstif_attr_dlg_t *ctx = pd->hid_ctx;

	if (ctx->creating)
		return;

	if ((reason == XmCR_RESIZE) || (pd->resized == 0))
		rnd_ltf_preview_zoom_update(pd);
	rnd_ltf_preview_redraw(pd);
}

void rnd_ltf_preview_invalidate(const rnd_box_t *screen)
{
	rnd_ltf_preview_t *prv;

	for(prv = gdl_first(&ltf_previews); prv != NULL; prv = prv->link.next) {
		if (!prv->redraw_with_design) continue;
		if (screen != NULL) {
			rnd_box_t pb;
			pb.X1 = prv->x1;
			pb.Y1 = prv->y1;
			pb.X2 = prv->x2;
			pb.Y2 = prv->y2;
			if (rnd_box_intersect(screen, &pb))
				rnd_ltf_preview_redraw(prv);
		}
		else
			rnd_ltf_preview_redraw(prv);
	}
}

void rnd_ltf_preview_add(rnd_ltf_preview_t *prv)
{
	gdl_insert(&ltf_previews, prv, link);
}

void rnd_ltf_preview_del(rnd_ltf_preview_t *prv)
{
	gdl_remove(&ltf_previews, prv, link);
}
