/*
A library to allow applictions to provide simple indications of
information to be displayed to users of the application through the
interface shell.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of either or both of the following licenses:

1) the GNU Lesser General Public License version 3, as published by the 
Free Software Foundation; and/or
2) the GNU Lesser General Public License version 2.1, as published by 
the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the applicable version of the GNU Lesser General Public 
License for more details.

You should have received a copy of both the GNU Lesser General Public 
License version 3 and version 2.1 along with this program.  If not, see 
<http://www.gnu.org/licenses/>
*/

#ifndef INDICATE_LISTENER_H_INCLUDED__
#define INDICATE_LISTENER_H_INCLUDED__ 1

#include <glib.h>
#include <glib-object.h>

#include "indicator.h"
#include "server.h"
#include "interests.h"

G_BEGIN_DECLS

/* Boilerplate */
#define INDICATE_TYPE_LISTENER (indicate_listener_get_type ())
#define INDICATE_LISTENER(object) (G_TYPE_CHECK_INSTANCE_CAST ((object), INDICATE_TYPE_LISTENER, IndicateListener))
#define INDICATE_IS_LISTENER(object) (G_TYPE_CHECK_INSTANCE_TYPE((object), INDICATE_TYPE_LISTENER))
#define INDICATE_LISTENER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST((klass), INDICATE_TYPE_LISTENER, IndicateListenerClass))
#define INDICATE_IS_LISTENER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), INDICATE_TYPE_LISTENER))
#define INDICATE_LISTENER_GET_CLASS(object) (G_TYPE_INSTANCE_GET_CLASS((object), INDICATE_TYPE_LISTENER, IndicateListenerClass))

#define INDICATE_LISTENER_SIGNAL_INDICATOR_ADDED       "indicator-added"
#define INDICATE_LISTENER_SIGNAL_INDICATOR_REMOVED     "indicator-removed"
#define INDICATE_LISTENER_SIGNAL_INDICATOR_MODIFIED    "indicator-modified"
#define INDICATE_LISTENER_SIGNAL_SERVER_ADDED          "server-added"
#define INDICATE_LISTENER_SIGNAL_SERVER_REMOVED        "server-removed"
#define INDICATE_LISTENER_SIGNAL_SERVER_COUNT_CHANGED  "server-count-changed"

#define INDICATE_LISTENER_SERVER_DBUS_NAME(server)   (indicate_listener_server_get_dbusname(server))
#define INDICATE_LISTENER_INDICATOR_ID(indicator)    (indicate_listener_indicator_get_id(indicator))

/**
 * IndicateListenerServer:
 *
 * A private structure used to identify a specific Indicate
 * server that exists on the bus.  Used as a paramaeter and a
 * signal parameter.
 */
typedef struct _IndicateListenerServer IndicateListenerServer;
#define INDICATE_TYPE_LISTENER_SERVER (indicate_listener_server_get_gtype ())
GType                   indicate_listener_server_get_gtype (void) G_GNUC_CONST;

/**
 * IndicateListenerIndicator:
 *
 * A private structure used to identify a specific Indicate
 * Indicator that exists on a #IndicateListenerServer.  Used as a
 * paramaeter and a signal parameter.
 */
typedef struct _IndicateListenerIndicator IndicateListenerIndicator;
#define INDICATE_TYPE_LISTENER_INDICATOR (indicate_listener_indicator_get_gtype ())
GType                   indicate_listener_indicator_get_gtype (void) G_GNUC_CONST;


typedef struct _IndicateListener IndicateListener;
struct _IndicateListener {
	GObject parent;
};


/**
 * IndicateListenerClass:
 * @parent: Parent class #GObjectClass.
 * @indicator_added: Slot for IndicateListener::indicator-added.
 * @indicator_removed: Slot for IndicateListener::indicator-removed.
 * @indicator_modified: Slot for IndicateListener::indicator-modified.
 * @server_added: Slot for IndicateListener::server-added.
 * @server_removed: Slot for IndicateListener::server-removed.
 * @server_count_changed: Slot for IndicateListener::server-count-changed.
 * @indicator_servers_report: Slot for IndicateListener::indicator-servers-report.
 * @indicate_listener_reserved1: Reserved for future use
 * @indicate_listener_reserved2: Reserved for future use
 * @indicate_listener_reserved3: Reserved for future use
 * 
 * The class has all of the signals that are supplied by
 * the listener about what is happening on the server.
*/
typedef struct _IndicateListenerClass IndicateListenerClass;
struct _IndicateListenerClass {
	/* Parent */
	GObjectClass parent;

	/* Signals */
	void (* indicator_added) (IndicateListenerServer * server, IndicateListenerIndicator * indicator);
	void (* indicator_removed) (IndicateListenerServer * server, IndicateListenerIndicator * indicator);
	void (* indicator_modified) (IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property);

	void (* server_added) (IndicateListenerServer * server, gchar * type);
	void (* server_removed) (IndicateListenerServer * server, gchar * type);
	void (* server_count_changed) (IndicateListenerServer * server, guint count);

	void (* indicator_servers_report) (void);

	/* Future Use */
	void (*indicate_listener_reserved1)(void);
	void (*indicate_listener_reserved2)(void);
	void (*indicate_listener_reserved3)(void);
};

GType indicate_listener_get_type (void) G_GNUC_CONST;

typedef void (*indicate_listener_get_property_variant_cb) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, GVariant * propertydata, gpointer data);
typedef void (*indicate_listener_get_property_cb) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, const gchar * propertydata, gpointer data);
typedef void (*indicate_listener_get_property_time_cb) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, const GTimeVal * propertydata, gpointer data);
typedef void (*indicate_listener_get_property_int_cb) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, gint propertydata, gpointer data);
typedef void (*indicate_listener_get_property_bool_cb) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, gboolean propertydata, gpointer data);
typedef void (*indicate_listener_get_server_property_cb) (IndicateListener * listener, IndicateListenerServer * server, const gchar * value, gpointer data);
typedef void (*indicate_listener_get_server_uint_property_cb) (IndicateListener * listener, IndicateListenerServer * server, guint value, gpointer data);

/* Create a new listener */
IndicateListener *    indicate_listener_new                (void);
IndicateListener *    indicate_listener_ref_default        (void);
void                  indicate_listener_get_property_variant (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            gchar * property,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, GVariant * propertydata, gpointer data),
                                                            gpointer data);
void                  indicate_listener_get_property       (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            gchar * property,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, const gchar * propertydata, gpointer data),
                                                            gpointer data);
void                  indicate_listener_get_property_time  (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            gchar * property,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, const GTimeVal * propertydata, gpointer data),
                                                            gpointer data);
void                  indicate_listener_get_property_int   (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            gchar * property,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, gint propertydata, gpointer data),
                                                            gpointer data);
void                  indicate_listener_get_property_bool  (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            gchar * property,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, gboolean propertydata, gpointer data),
                                                            gpointer data);
GList *               indicate_listener_get_servers        (IndicateListener * listener);
void                  indicate_listener_display            (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            guint timestamp);
void                  indicate_listener_displayed          (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            IndicateListenerIndicator * indicator,
                                                            gboolean displayed);
void                  indicate_listener_server_get_type    (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, const gchar * value, gpointer data),
                                                            gpointer data);
void                  indicate_listener_server_get_desktop (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, const gchar * value, gpointer data),
                                                            gpointer data);
void                  indicate_listener_server_get_count   (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, guint value, gpointer data),
                                                            gpointer data);
void                  indicate_listener_server_get_menu    (IndicateListener * listener,
                                                            IndicateListenerServer * server,
                                                            void (*callback) (IndicateListener * listener, IndicateListenerServer * server, const gchar * value, gpointer data),
                                                            gpointer data);
void                  indicate_listener_server_get_icon_theme (IndicateListener * listener,
							       IndicateListenerServer * server,
							       void (*callback) (IndicateListener *listener, IndicateListenerServer *server, const gchar *value, gpointer data),
							       gpointer data);
GList *               indicate_listener_server_get_indicators    (IndicateListener * listener,
                                                                  IndicateListenerServer * server);
const gchar *         indicate_listener_server_get_dbusname      (IndicateListenerServer * server);
const gchar *         indicate_listener_server_get_dbuspath      (IndicateListenerServer *server);
guint                 indicate_listener_indicator_get_id         (IndicateListenerIndicator * indicator);
void                  indicate_listener_server_show_interest     (IndicateListener * listener,
                                                                  IndicateListenerServer * server,
                                                                  IndicateInterests interest);
void                  indicate_listener_server_remove_interest   (IndicateListener * listener,
                                                                  IndicateListenerServer * server,
                                                                  IndicateInterests interest);
gboolean              indicate_listener_server_check_interest    (IndicateListener * listener,
                                                                  IndicateListenerServer * server,
                                                                  IndicateInterests interest);
void                  indicate_listener_set_default_max_indicators (IndicateListener * listener,
                                                                  gint max);
void                  indicate_listener_set_server_max_indicators    (IndicateListener * listener,
                                                                  IndicateListenerServer * server,
                                                                  gint max);

G_END_DECLS

#endif /* INDICATE_LISTENER_H_INCLUDED__ */

