/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <stdbool.h>

#include "cdw_cdll.h"
#include "cdw_debug.h"


/**
 * \file cdw_cdll.c
 * \brief Bug-ridden implementation of doubly circularly linked list
 *
 */




static bool        cdw_cdll_is_valid_head(cdw_cdll_t *head);
static bool        cdw_cdll_is_member(cdw_cdll_t *head, void *data);
static cdw_rv_t    cdw_cdll_append_non_unique(cdw_cdll_t *head, void *data);
static cdw_cdll_t *cdw_cdll_last_element(cdw_cdll_t *head);
static cdw_cdll_t *cdw_cdll_get_ith_element(cdw_cdll_t *head, size_t i);
static cdw_cdll_t *cdw_cdll_new_element(void);




/**
 * \brief Initialize a list
 *
 * Call this function when you create new list. 'head' parameter will be
 * used to reference the list and it is the only element of list that
 * needs to be initialized by this function.
 *
 * The function doesn't put any user data into this element - it just
 * initializes the list.
 *
 * \param head - first element of list
 *
 * \return CDW_ERROR if some malloc error occurs
 * \return CDW_NO if list is already initialized
 * \return CDW_OK if success
 */
cdw_rv_t cdw_cdll_init(cdw_cdll_t **head)
{
	cdw_assert (*head == (cdw_cdll_t *) NULL, "passing non-null head to init()\n");

	*head = cdw_cdll_new_element();

	if (*head == NULL) {
		return CDW_ERROR;
	} else {
		(*head)->prev = *head;
		(*head)->next = *head;
		(*head)->data = (void *) NULL;
		return CDW_OK;
	}
}





/**
 * \brief Returns last element of list
 *
 * Useful when you want to append new data at the end of list.
 * "Last" in this implementation of doubly-linked list is defined as
 * "last one that you meet when starting from head, going forward, and
 * still not getting back to head"
 *
 * \return NULL if list is empty
 * \return pointer to last element on list if list is not empty
 */
cdw_cdll_t *cdw_cdll_last_element(cdw_cdll_t *head)
{
	cdw_assert (cdw_cdll_is_valid_head(head), "head is invalid\n");

	return head->prev;
}





/**
 * \brief Allocate new list element
 *
 * Pointers to previous and next list elements are initialized to
 * (cdw_cdll_t *) NULL. Pointer to data is initialized to (void *) NULL.
 *
 * \return Pointer to freshly allocated list element
 * \return NULL if malloc fails
 */
cdw_cdll_t *cdw_cdll_new_element(void)
{
	cdw_cdll_t *e = (cdw_cdll_t *) malloc(sizeof(cdw_cdll_t));
	if (e == NULL) {
		return (cdw_cdll_t *) NULL;
	}

	e->data = (void *) NULL;
	e->next = (cdw_cdll_t *) NULL;
	e->prev = (cdw_cdll_t *) NULL;

	return e;
}





/**
 * \brief Append data (in new list element) at the end of list
 *
 * The function does not check if given data already exists on list.
 * In order to avoid duplicates on list you have to search for given
 * data on the list first.
 *
 * The function will initialize list if list is empty.
 *
 * \param data - pointer to data that you want to append to list
 *
 * \return CDW_ERROR if malloc fails
 * \return CDW_OK if success
 */
cdw_rv_t cdw_cdll_append_non_unique(cdw_cdll_t *head, void *data)
{
	cdw_assert (cdw_cdll_is_valid_head(head), "head of list is invalid\n");

	if (cdw_cdll_is_empty(head)) {
		head->data = data;
		return CDW_OK;
	} else {
		cdw_cdll_t *last = cdw_cdll_last_element(head);
		if (last == (cdw_cdll_t *) NULL) {
			return CDW_ERROR;
		} else {
			cdw_cdll_t *e = cdw_cdll_new_element();
			if (e == (cdw_cdll_t *) NULL) {
				return CDW_ERROR;
			} else {
				head->prev = e;
				e->next = head;

				last->next = e;
				e->prev = last;

				e->data = data;
				return CDW_OK;
			}
		}
	}
}





/**
 * \brief Append pointer to data (in new list element) at the end of list
 *
 * The function checks if given data already exists on list. Item
 * that would be duplicate on the list is not appended and CDW_NO is returned.
 *
 * \param data - pointer to data that you want to store on list
 *
 * \return CDW_ERROR if malloc fails
 * \return CDW_NO if data is already on list
 * \return CDW_OK if success
 */
cdw_rv_t cdw_cdll_append(cdw_cdll_t *head, void *data)
{
	if (cdw_cdll_is_member(head, data)) {
		return CDW_NO;
	} else {
		/* will init list if necessary */
		return cdw_cdll_append_non_unique(head, data);
	}
}




/**
 * \brief Return pointer to i-th element on filelist
 *
 * \param i - index of list element containing data that you want to obtain (0-based)
 *
 * \return Pointer to i-th element of list
 * \return NULL if index is too large or list is empty
 */
cdw_cdll_t *cdw_cdll_get_ith_element(cdw_cdll_t *head, size_t i)
{
	if (head == (cdw_cdll_t *) NULL) {
		return NULL;
	}

	if (head->next == head) {
		return head;
	}

	cdw_cdll_t *elem = head;
	size_t j = 0;


	for (j = 0; j <= i; j++) {
		if (j == i) {
			return elem;
		} else {
			elem = elem->next;
		}
	}

	return NULL;
}





/**
 * \brief Return pointer to data in i-th element of list
 *
 * \param i - index of list element containing data that you want to obtain (0-based)
 *
 * \return Pointer to i-th element of list
 * \return NULL if index is too large or list is empty
 */
void *cdw_cdll_ith_data(cdw_cdll_t *head, size_t i)
{
	if (head == (cdw_cdll_t *) NULL) {
		return NULL;
	}

	cdw_cdll_t *element = cdw_cdll_get_ith_element(head, i);

	if (element == NULL) {
		return NULL;
	} else {
		return element->data;
	}
}





/**
 * \brief Remove i-th element from list
 *
 * \param i - index of element that should be removed (zero-based)
 *
 * \return CDW_ERROR if list is empty or other error occurred
 * \return CDW_OK if success
 */
cdw_rv_t cdw_cdll_remove(cdw_cdll_t **head, size_t i)
{
	if (*head == (cdw_cdll_t *) NULL) {
		return CDW_ERROR;
	}

	cdw_cdll_t *f = cdw_cdll_get_ith_element(*head, i);

	if (f == NULL) {
		return CDW_ERROR;
	}

	/* works even if j is first or last node, NULLs are preserved */
	if (f->next != NULL) {
		f->next->prev = f->prev;
	}
	if (f->prev != NULL) {
		f->prev->next = f->next;
	}


	if (f == *head) {
		if (f->next == *head) {
			/* this was the only element on list; pointer to
			   data can be NULLed, but otherwise head must be
			   intact */
			f->data = (void *) NULL;
		} else {
			/* we have to create new list's head */

			*head = f->next;

			f->data = (void *) NULL;
			free(f);
			f = (cdw_cdll_t *) NULL;
		}
	}



	return CDW_OK;
}





/**
 * \brief Count files on list
 *
 * \return Number of files on filelist (0 if no files)
 */
size_t cdw_cdll_length(cdw_cdll_t *head)
{
	if (cdw_cdll_is_empty(head)) {
		return 0;
	} else {
		cdw_cdll_t *f;
		size_t i = 1;

		for (f = head; f->next != head; f = f->next) {
			i++;
		}
		return i;
	}
}




/**
 * \brief Remove all elements from list
 *
 * \return CDW_OK
 */
cdw_rv_t cdw_cdll_destroy(cdw_cdll_t *head)
{
	size_t len = cdw_cdll_length(head);
	for (size_t i = 0; i < len; i++) {
		cdw_cdll_t *tmp = head->next;
		cdw_sdm ("INFO: free %zd\n", i);
		free(head);
		head = tmp;
	}

	return CDW_OK;
}





/**
 * \brief Check if list is empty
 *
 * You could use cdw_cdll_elements() and check if its return value is
 * zero, but cdw_cdll_is_empty() is faster when calling it for long,
 * non-empty lists.
 *
 * List is treated as invalid (and thus as empty) if its first element is
 * invalid: when 'next' or 'prev' or 'data' pointer is NULL.
 *
 *
 *
 * \return true if list is empty or invalid
 * \return false if list is not empty
 */
bool cdw_cdll_is_empty(cdw_cdll_t *head)
{
	cdw_assert (cdw_cdll_is_valid_head(head), "head of list is invalid\n");

	if (head->data == (void *) NULL) {
		return true;
	} else {
		return false;
	}
}





bool cdw_cdll_is_valid_head(cdw_cdll_t *head)
{
	if (head != (cdw_cdll_t *) NULL
		   && head->next != (cdw_cdll_t *) NULL
		   && head->prev != (cdw_cdll_t *) NULL) {

		return true;
	} else {
		return false;
	}
}




/**
 * \brief Check if given pointer to data is already on the list
 *
 * Search for given pointer to data on the list. Return true if
 * the pointer is on the list, return false if the pointer is not on
 * the list.
 *
 * The function also works for empty list: it returns false.
 *
 * \param data - pointer to data which you want to look for on the list
 *
 * \return true if given pointer is on the list
 * \return false if given pointer is not on the list (or the list is empty)
 */
bool cdw_cdll_is_member(cdw_cdll_t *head, void *data)
{
	cdw_assert (cdw_cdll_is_valid_head(head), "head of list is invalid\n");
	if (cdw_cdll_is_empty(head)) {
		return false;
	}

	if (data == (cdw_cdll_t *) NULL) {
		return false;
	}

	if (head->next == head) {
		if (head->data == data) {
			return true;
		}
	}

	cdw_cdll_t *e;
	for (e = head; e->next != head; e = e->next) {
		if (e->data == data) {
			return true;
		}
	}

	return false;
}



#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */

#include <string.h>

void test_cdw_cdll_new_element(void);
void test_cdw_cdll_init(void);
void test_cdw_cdll_append(void);
void test_cdw_cdll_is_empty(void);
void test_cdw_cdll_length(void);
void test_cdw_cdll_remove(void);



void cdw_cdll_run_tests(void)
{
	fprintf(stderr, "testing cdw_cdll.c\n");

	test_cdw_cdll_new_element();
	test_cdw_cdll_init();
	test_cdw_cdll_is_empty();
	test_cdw_cdll_append();
	test_cdw_cdll_length();
	test_cdw_cdll_remove();

	fprintf(stderr, "done\n\n");

	return;
}




void test_cdw_cdll_new_element(void)
{
	fprintf(stderr, "\ttesting cdw_cdll_new_element()... ");

	cdw_cdll_t *head = cdw_cdll_new_element();

	cdw_assert_test (head != (cdw_cdll_t *) NULL, "head == NULL");

	cdw_assert_test (head->next == (cdw_cdll_t *) NULL, "next != NULL");
	cdw_assert_test (head->prev == (cdw_cdll_t *) NULL, "prev != NULL");
	cdw_assert_test (head->data == (void *) NULL, "data != NULL");

	free(head);

	fprintf(stderr, "OK\n");

	return;
}




void test_cdw_cdll_init(void)
{
	fprintf(stderr, "\ttesting cdw_cdll_init()... ");

	cdw_cdll_t *head = NULL;
	cdw_rv_t crv = cdw_cdll_init(&head);

	assert (crv == CDW_OK);

	assert(head->next == head);
	assert(head->prev == head);
	assert(head->data == (void *) NULL);

	crv = cdw_cdll_destroy(head);
	assert (crv == CDW_OK);

	fprintf(stderr, "OK\n");

	return;
}




void test_cdw_cdll_is_empty(void)
{
	fprintf(stderr, "\ttesting cdw_cdll_is_empty()... ");

	bool is_empty = false;

	/* cdw_cdll_is_empty() can only work for valid list head, so
	   first make valid head, and then only modify data pointer */
	cdw_cdll_t *head = (cdw_cdll_t *) NULL;
	cdw_cdll_init(&head);

	is_empty = cdw_cdll_is_empty(head);
	assert (is_empty);

	head->data = malloc(10);
	assert (head->data != (void *) NULL);

	is_empty = cdw_cdll_is_empty(head);
	assert (!is_empty);

	free(head->data);

	cdw_rv_t crv = cdw_cdll_destroy(head);
	assert(crv == CDW_OK);

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_cdll_append(void)
{
	fprintf(stderr, "\ttesting cdw_cdll_append()... ");

	cdw_cdll_t *head = (cdw_cdll_t *) NULL;
	cdw_cdll_init(&head);

	struct my_data_t {
		char string[15];
		int number;
	};

	struct my_data_t *retrieved = (struct my_data_t *) NULL;
	int l = 0;
	cdw_rv_t crv = CDW_NO;


	/* *** appending first item *** */

	struct my_data_t my_data;
	strncpy(my_data.string, "hello world", 12);
	my_data.number = 11;

	crv = cdw_cdll_append(head, (void *) &my_data);
	assert (crv == CDW_OK);

	/* appending data to empty list should not modify links */
	cdw_assert_test (head->next == head, "invalid next (1)");
	cdw_assert_test (head->prev == head, "invalid prev (1)");

	retrieved = head->data;

	l = strcmp(retrieved->string, my_data.string);
	cdw_assert_test (l == 0, "head string not appended (1)");
	cdw_assert_test (retrieved->number == my_data.number, "head number not appended (1)");



	/* *** append second item *** */
	struct my_data_t my_data2;
	strncpy(my_data2.string, "asteroid", 10);
	my_data2.number = 22;

	crv = cdw_cdll_append(head, (void *) &my_data2);
	assert (crv == CDW_OK);

	/* first check links */
	cdw_assert_test (head->next->next == head, "invalid next (2)");
	cdw_assert_test (head->prev->prev == head, "invalid prev (2)");
	cdw_assert_test (head->prev == head->next, "invalid circle (2)");

	/* re-check data from head */
	retrieved = head->data;
	l = strcmp(retrieved->string, my_data.string);
	cdw_assert_test (l == 0, "head string corrupted (2)");
	cdw_assert_test (retrieved->number == my_data.number, "head number corrupted (2)");


	/* check data from second element of list */
	retrieved = head->next->data;
	l = strcmp(retrieved->string, my_data2.string);
	cdw_assert_test (l == 0, "second element string not appended (2)");
	cdw_assert_test (retrieved->number == my_data2.number, "second element number not appended (2)");



	/* *** append third item *** */
	struct my_data_t my_data3;
	strncpy(my_data2.string, "melbourne", 10);
	my_data2.number = 33;

	crv = cdw_cdll_append(head, (void *) &my_data3);
	assert (crv == CDW_OK);

	/* first check links */
	cdw_assert_test (head->next->next->next == head, "invalid next (3)");
	cdw_assert_test (head->prev->prev->prev == head, "invalid prev (3)");
	cdw_assert_test (head->prev->prev == head->next, "invalid circle (3a)");
	cdw_assert_test (head->prev == head->next->next, "invalid circle (3b)");

	/* re-check data from head */
	retrieved = head->data;
	l = strcmp(retrieved->string, my_data.string);
	cdw_assert_test (l == 0, "head string corrupted (3)");
	cdw_assert_test (retrieved->number == my_data.number, "head number corrupted (3)");

	/* check data from second element of list */
	retrieved = head->next->data;
	l = strcmp(retrieved->string, my_data2.string);
	cdw_assert_test (l == 0, "second element string not appended (3)");
	cdw_assert_test (retrieved->number == my_data2.number, "second element number not appended (3)");

	/* check data from third element of list */
	retrieved = head->next->next->data;
	l = strcmp(retrieved->string, my_data3.string);
	cdw_assert_test (l == 0, "third element string not appended (3)");
	cdw_assert_test (retrieved->number == my_data3.number, "third element number not appended (3)");



	crv = cdw_cdll_destroy(head);
	assert(crv == CDW_OK);

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_cdll_length(void)
{
	fprintf(stderr, "\ttesting cdw_cdll_length()... ");

	cdw_cdll_t *head = (cdw_cdll_t *) NULL;
	cdw_cdll_init(&head);

	int data1 = 123456;
	int data2 = 886644;
	int data3 = 13579;
	int data4 = 24680;
	int data5 = 123612;
	int data6 = 98765;

	size_t len = cdw_cdll_length(head);
	cdw_assert_test (len == 0, "len is not 0\n");

	cdw_rv_t crv = cdw_cdll_append(head, &data1);
	cdw_assert_test (crv == CDW_OK, "failed to append first element\n");
	len = cdw_cdll_length(head);
	cdw_assert_test (len == 1, "len is not 1\n");

	crv = cdw_cdll_append(head, &data2);
	cdw_assert_test (crv == CDW_OK, "failed to append second element\n");
	len = cdw_cdll_length(head);
	cdw_assert_test (len == 2, "len is not 2\n");

	crv = cdw_cdll_append(head, &data3);
	cdw_assert_test (crv == CDW_OK, "failed to append third element\n");
	len = cdw_cdll_length(head);
	cdw_assert_test (len == 3, "len is not 3\n");

	crv = cdw_cdll_append(head, &data4);
	cdw_assert_test (crv == CDW_OK, "failed to append fourth element\n");
	len = cdw_cdll_length(head);
	cdw_assert_test (len == 4, "len is not 4\n");

	crv = cdw_cdll_append(head, &data5);
	cdw_assert_test (crv == CDW_OK, "failed to append fifth element\n");
	len = cdw_cdll_length(head);
	cdw_assert_test (len == 5, "len is not 5\n");

	crv = cdw_cdll_append(head, &data6);
	cdw_assert_test (crv == CDW_OK, "failed to append sixth element\n");
	len = cdw_cdll_length(head);
	cdw_assert_test (len == 6, "len is not 6\n");

	fprintf(stderr, "OK\n");
	return;
}






void test_cdw_cdll_remove(void)
{
	fprintf(stderr, "\ttesting cdw_cdll_remove()... ");

	cdw_cdll_t *head = (cdw_cdll_t *) NULL;
	cdw_cdll_init(&head);

	int data0 = 123456;
	int data1 = 886644;
	int data2 = 13579;
	int data3 = 24680;
	int data4 = 123612;
	int data5 = 98765;

	int *retrieved;

	size_t len = cdw_cdll_length(head);
	cdw_assert_test (len == 0, "len is not 0\n");

	cdw_rv_t crv = cdw_cdll_append(head, &data0);
	cdw_assert_test (crv == CDW_OK, "failed to append first element\n");
	retrieved = head->data;
	cdw_assert_test (data0 == *retrieved, "incorrect data (0)\n");

	crv = cdw_cdll_append(head, &data1);
	cdw_assert_test (crv == CDW_OK, "failed to append second element\n");
	retrieved = head->next->data;
	cdw_assert_test (data1 == *retrieved, "incorrect data (1)\n");

	crv = cdw_cdll_append(head, &data2);
	cdw_assert_test (crv == CDW_OK, "failed to append third element\n");
	retrieved = head->next->next->data;
	cdw_assert_test (data2 == *retrieved, "incorrect data (2)\n");

	crv = cdw_cdll_append(head, &data3);
	cdw_assert_test (crv == CDW_OK, "failed to append fourth element\n");
	retrieved = head->next->next->next->data;
	cdw_assert_test (data3 == *retrieved, "incorrect data (3)\n");

	crv = cdw_cdll_append(head, &data4);
	cdw_assert_test (crv == CDW_OK, "failed to append fifth element\n");
	retrieved = head->next->next->next->next->data;
	cdw_assert_test (data4 == *retrieved, "incorrect data (4)\n");

	crv = cdw_cdll_append(head, &data5);
	cdw_assert_test (crv == CDW_OK, "failed to append sixth element\n");
	retrieved = head->next->next->next->next->next->data;
	cdw_assert_test (data5 == *retrieved, "incorrect data (5)\n");

	len = cdw_cdll_length(head);
	cdw_assert_test (len == 6, "len is not 6\n");

	/* at this point data is prepared and checked, ready for removing */

	/* first remove - last element */

	crv = cdw_cdll_remove(&head, 5);

	len = cdw_cdll_length(head);
	cdw_assert_test (len == 5, "len is not 5\n");

	cdw_assert_test (head->next->next->next->next->next->data == head->data, "circle not closed\n");
	cdw_assert_test (head->prev->prev->prev->prev->prev->data == head->data, "circle not closed (2)\n");

	retrieved = head->data;
	cdw_assert_test (*retrieved == data0, "data 0 invalid after remove\n");

	retrieved = head->prev->data;
	cdw_assert_test (*retrieved == data4, "data 4 invalid after remove\n");

	/* second remove - element in middle */

	crv = cdw_cdll_remove(&head, 2);

	len = cdw_cdll_length(head);
	cdw_assert_test (len == 4, "len is not 4\n");

	cdw_assert_test (head->next->next->next->next->data == head->data, "circle not closed (3)\n");
	cdw_assert_test (head->prev->prev->prev->prev->data == head->data, "circle not closed (4)\n");

	retrieved = head->data;
	cdw_assert_test (*retrieved == data0, "data 0 invalid after remove\n");

	retrieved = head->next->data;
	cdw_assert_test (*retrieved == data1, "data 1 invalid after remove\n");

	retrieved = head->next->next->data;
	cdw_assert_test (*retrieved == data3, "data 3 invalid after remove\n");

	retrieved = head->next->next->next->data;
	cdw_assert_test (*retrieved == data4, "data 4 invalid after remove\n");

	retrieved = head->next->next->next->next->data;
	cdw_assert_test (*retrieved == data0, "data 0 invalid after remove\n");

	retrieved = head->data;
	cdw_assert_test (*retrieved == data0, "data 0 invalid after remove\n");

	retrieved = head->prev->data;
	cdw_assert_test (*retrieved == data4, "data 4 invalid after remove\n");

	retrieved = head->prev->prev->data;
	cdw_assert_test (*retrieved == data3, "data 3 invalid after remove\n");

	retrieved = head->prev->prev->prev->data;
	cdw_assert_test (*retrieved == data1, "data 1 invalid after remove\n");

	retrieved = head->prev->prev->prev->prev->data;
	cdw_assert_test (*retrieved == data0, "data 0 invalid after remove\n");

	/* third remove - removing first element of list - head will be updated */

	crv = cdw_cdll_remove(&head, 0);

	len = cdw_cdll_length(head);
	cdw_assert_test (len == 3, "len is not 3\n");

	cdw_assert_test (head->next->next->next->data == head->data, "circle not closed (3)\n");
	cdw_assert_test (head->prev->prev->prev->data == head->data, "circle not closed (4)\n");

	retrieved = head->data;
	cdw_assert_test (*retrieved == data1, "data 1 invalid after remove\n");

	retrieved = head->next->data;
	cdw_assert_test (*retrieved == data3, "data 3 invalid after remove\n");

	retrieved = head->next->next->data;
	cdw_assert_test (*retrieved == data4, "data 4 invalid after remove\n");

	retrieved = head->prev->data;
	cdw_assert_test (*retrieved == data4, "data 4 invalid after remove\n");

	retrieved = head->prev->prev->data;
	cdw_assert_test (*retrieved == data3, "data 3 invalid after remove\n");


	fprintf(stderr, "OK\n");
	return;
}


#endif /* #ifdef CDW_UNIT_TEST_CODE */
