/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _GNU_SOURCE /* asprintf() */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdbool.h>

#include "gettext.h"
#include "cdw_string.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_debug.h"
#include "cdw_tabs_window.h"


/**
   \file cdw_tabs_file.c

   File implements "tabbed window" widget: a window with N separate
   overlapping (stacked) areas, i.e. tabs or panels.

   Each tab provides a window and subwindow. Code using this widget
   can add any content to the tab using the window and subwindow.
*/




static bool cdw_tabs_window_is_hotkey(cdw_tabs_window_t *twindow, int key, int *ti);
static bool cdw_tabs_window_is_return_key(cdw_tabs_window_t *twindow, int key);
static void cdw_tabs_window_refresh_current_tab(cdw_tabs_window_t *twindow);




/**
   \brief Initialize data structure for tabbed window

   The function returns a pointer to data structure representing tabbed
   window, to which you can append (attach) tabs. After attaching all
   tabs you still need to finalize the window, and then you are ready
   to call driver function for this window.

   Number of tabs attached to the window must exactly match value
   of \p n_tabs at the time you finalize the tabbed window.
   You can't append more tabs than \p n_tabs, and you can't leave
   adding some tabs "for later".

   \param n_tabs - number of tabs that will be visible in the window
   \param title - title of the window, common for all tabs
   \param subtitle - string displayed at the bottom of the window, common for all tabs

   \return pointer to initial tabbed window on success
   \return NULL pointer on failure
*/
cdw_tabs_window_t *cdw_tabs_window_init(int n_tabs, const char *title, const char *subtitle)
{
	cdw_tabs_window_t *twindow = (cdw_tabs_window_t *) malloc(sizeof (cdw_tabs_window_t));
	if (twindow == (cdw_tabs_window_t *) NULL) {
		cdw_vdm ("ERROR: failed to create tabs window\n");
		return (cdw_tabs_window_t *) NULL;
	}

	twindow->tabs = (cdw_tab_t *) NULL;
	twindow->n_tabs = 0;
	twindow->current_tab = 0;

	twindow->title = (char *) NULL;
	twindow->subtitle = (char *) NULL;

	twindow->user_data = (void *) NULL;

	twindow->geometry.n_lines = 0;
	twindow->geometry.n_cols = 0;
	twindow->geometry.begin_y = 0;
	twindow->geometry.begin_x = 0;
	twindow->geometry.tabs_width = 0;

	twindow->tabs = (cdw_tab_t *) malloc((long unsigned int) n_tabs * sizeof (cdw_tab_t));
	if (twindow->tabs == (cdw_tab_t *) NULL) {
		cdw_vdm ("ERROR: failed to create tabs in twindow\n");
		free(twindow);
		twindow = (cdw_tabs_window_t *) NULL;
		return (cdw_tabs_window_t *) NULL;
	}

	twindow->title = strdup(title);
	twindow->subtitle = strdup(subtitle);
	if (twindow->title == (char *) NULL || twindow->subtitle == (char *) NULL) {
		cdw_vdm ("ERROR: failed to strdup window title \"%s\" or subtitle \"%s\"\n",
			 title, subtitle);
		cdw_tabs_window_delete(&twindow);
		return (cdw_tabs_window_t *) NULL;
	}

	twindow->n_tabs = n_tabs;

	for (int i = 0; i < twindow->n_tabs; i++) {
		twindow->tabs[i].label = (char *) NULL;
		twindow->tabs[i].id = 0;
		twindow->tabs[i].hotkey = 0;

		twindow->tabs[i].panel = (PANEL *) NULL;
		twindow->tabs[i].window = (WINDOW *) NULL;
		twindow->tabs[i].subwindow = (WINDOW *) NULL;
	}

	twindow->n_return_keys = 0;
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		twindow->return_keys[i] = 0;
	}

	return twindow;;
}





cdw_rv_t cdw_tabs_window_append_tab(cdw_tabs_window_t *twindow, cdw_id_t id, int hotkey, const char *label)
{
	cdw_assert (twindow != (cdw_tabs_window_t *) NULL, "ERROR: twindow is NULL\n");
	cdw_assert (label != (char *) NULL, "ERROR: tab label is NULL\n");

	/* search for first free slot and "insert" a new tab there */
	for (int i = 0; i < twindow->n_tabs; i++) {
		if (twindow->tabs[i].label == (char *) NULL) {
			twindow->tabs[i].id = id;
			asprintf(&(twindow->tabs[i].label), "<%s>%s", cdw_ncurses_key_label(hotkey), label);
			cdw_vdm ("INFO: added label #%d: \"%s\"\n", i, twindow->tabs[i].label);
			twindow->tabs[i].hotkey = hotkey;
			return CDW_OK;
		}
	}

	cdw_vdm ("ERROR: trying to append tab, but no space left\n");
	return CDW_ERROR;
}





/**
   \brief Set geometry constraints of given tabbed window

   \param twindow - window for which to set geometry constraints
   \param n_lines - number of lines (rows) of window
   \param n_cols - number of columns in window
   \param begin_y - Y coordinate of window (relative to 0,0 point of screen)
   \param begin_x - X coordinate of window (relative to 0,0 point of screen)
   \param tabs_width - number of columns occupied by tabs with labels on right side of window
*/
void cdw_tabs_window_set_geometry(cdw_tabs_window_t *twindow, int n_lines, int n_cols, int begin_y, int begin_x, int tabs_width)
{
	twindow->geometry.n_lines = n_lines;
	twindow->geometry.n_cols = n_cols;
	twindow->geometry.begin_y = begin_y;
	twindow->geometry.begin_x = begin_x;
	twindow->geometry.tabs_width = tabs_width;

	return;
}





/**
   \brief Draw tabs that are visible on right side of tabbed window

   Function draws tabs (with labels) visible on right side of window,
   that make it easier to navigate tabbed window.

   \param twindow - tabbed window to draw in
*/
void cdw_tabs_window_draw_tabs(cdw_tabs_window_t *twindow)
{
	for (int i = 0; i < twindow->n_tabs; i++) {
		WINDOW *window = twindow->tabs[i].subwindow;
		/* this may happen if there are less tabs attached than n_tabs */
		cdw_assert (window != (WINDOW *) NULL, "ERROR: subwindow #%d is NULL\n", i);

		int start_col = twindow->geometry.n_cols - twindow->geometry.tabs_width;
		mvwvline(window, 0, start_col, ACS_VLINE, twindow->geometry.n_cols);

		/* draw horizontal lines and tab labels */
		for (int j = 0; j < twindow->n_tabs; j++) {
			mvwprintw(window, 2 * j, start_col + 1, "%s", twindow->tabs[j].label);
			mvwhline(window, 2 * j + 1, start_col + 1, ACS_HLINE, twindow->geometry.n_cols);
		}

		/* draw corners, specific for values of tab_number */
		if (i == 0) {
			mvwaddch(window, 0, start_col, ' ');
			mvwaddch(window, 1, start_col, ACS_ULCORNER);
		} else {
			mvwaddch(window, (i * 2) - 1, start_col, ACS_LLCORNER);
			mvwaddch(window, (i * 2), start_col, ' ');
			mvwaddch(window, (i * 2) + 1, start_col, ACS_ULCORNER);
		}
		wrefresh(window);
	}

	return;
}





/**
   \brief Finish creating and setting up all data necessary to run a tabbed window

   Call this function after calling these functions:
    - cdw_tabs_window_init()
    - cdw_tabs_window_set_geometry()
    - cdw_tabs_window_append_tab() (all necessary calls)
    and before adding any content to windows in tabbed window.

   \param twindow - tabbed window which should be finalized

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_tabs_window_finalize(cdw_tabs_window_t *twindow)
{
	for (int i = 0; i < twindow->n_tabs; i++) {
		twindow->tabs[i].window = cdw_window_new((WINDOW *) NULL,
							 twindow->geometry.n_lines,
							 twindow->geometry.n_cols,
							 (LINES - twindow->geometry.n_lines) / 2,
							 (COLS - twindow->geometry.n_cols) / 2,
							 CDW_COLORS_DIALOG, twindow->title, twindow->subtitle);
		if (twindow->tabs[i].window == (WINDOW *) NULL) {
			cdw_vdm ("ERROR: failed to create window #%d\n", i);
			return CDW_ERROR;
		}

		twindow->tabs[i].subwindow = cdw_window_new(twindow->tabs[i].window,
							    twindow->geometry.n_lines - 2,
							    twindow->geometry.n_cols - 2, 1, 1,
							    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);
		if (twindow->tabs[i].subwindow == (WINDOW *) NULL) {
			cdw_vdm ("ERROR: failed to create subwindow #%d\n", i);
			return CDW_ERROR;
		}

		twindow->tabs[i].panel = new_panel(twindow->tabs[i].window);
		if (twindow->tabs[i].panel == (PANEL *) NULL) {
			cdw_vdm ("ERROR: failed to create panel #%d with new_panel()\n", i);
			return CDW_ERROR;
		}

		keypad(twindow->tabs[i].subwindow, TRUE);
		nodelay(twindow->tabs[i].subwindow, FALSE);
		keypad(twindow->tabs[i].window, TRUE);
		nodelay(twindow->tabs[i].window, FALSE);

		redrawwin(twindow->tabs[i].subwindow);
		wrefresh(twindow->tabs[i].subwindow);
		redrawwin(twindow->tabs[i].window);
		wrefresh(twindow->tabs[i].window);
	}

	cdw_tabs_window_draw_tabs(twindow);

	return CDW_OK;
}





/**
   \brief Driver function for tabbed window widget

   The function handles switching between the tabs in tabbed window,
   and then passes control to tab-specific driver function. If the
   driver returns a return key (i.e. key configured with
   cdw_tabs_window_add_return_key()), the function returns the key value.

   \param twindow - tabbed window to be controlled

   \return last key pressed in tabbed window
*/
int cdw_tabs_window_driver(cdw_tabs_window_t *twindow)
{
	int key = twindow->tabs[twindow->current_tab].hotkey; /* safe default - switch to first tab */

	do {
		if (cdw_tabs_window_is_hotkey(twindow, key, &(twindow->current_tab))) {

			show_panel(twindow->tabs[twindow->current_tab].panel);
			update_panels();
			cdw_main_ui_main_window_wrefresh();
			cdw_tabs_window_refresh_current_tab(twindow);

		} else if (cdw_tabs_window_is_return_key(twindow, key)) {
			return key;
		} else {
			/* loop */
		}

		/* wgetch() may be replaced by tab-related driver;
		   in the most simple model (tabs contain only pure
		   windows) wgetch() is sufficient */
	} while ((key = twindow->driver_reader(twindow)));

	return CDW_CANCEL;
}





/**
   \brief Look up given key on list of hotkeys in given tabbed window

   \param twindow - tabbed window to examine
   \param key - key to look up on list of hotkeys for given twindow
   \param ti - place for index of tab, for which given key is a hotkey

   \return true if \p key is a hotkey for one of tabs in tabbed window
   \return false if \p key is not a hotkey for any of tabs in tabbed window
*/
bool cdw_tabs_window_is_hotkey(cdw_tabs_window_t *twindow, int key, int *ti)
{
	for (int i = 0; i < twindow->n_tabs; i++) {
		if (key == twindow->tabs[i].hotkey) {
			*ti = i;
			return true;
		}
	}
	return false;
}





void cdw_tabs_window_delete(cdw_tabs_window_t **twindow)
{
	if (*twindow == (cdw_tabs_window_t *) NULL) {
		cdw_vdm ("WARNING: passed NULL twindow to the function\n");
		return;
	}

	if ((*twindow)->title != (char *) NULL) {
		free((*twindow)->title);
		(*twindow)->title = (char *) NULL;
	}
	if ((*twindow)->subtitle != (char *) NULL) {
		free((*twindow)->subtitle);
		(*twindow)->subtitle = (char *) NULL;
	}

	if ((*twindow)->tabs != (cdw_tab_t *) NULL) {

		for (int i = 0; i < (*twindow)->n_tabs; i++) {

			if ((*twindow)->tabs[i].panel != (PANEL *) NULL) {
				del_panel((*twindow)->tabs[i].panel);
				(*twindow)->tabs[i].panel = (PANEL *) NULL;
			}

			if ((*twindow)->tabs[i].subwindow != (WINDOW *) NULL) {
				delwin((*twindow)->tabs[i].subwindow);
				(*twindow)->tabs[i].subwindow = (WINDOW *) NULL;
			}

			if ((*twindow)->tabs[i].window != (WINDOW *) NULL) {
				delwin((*twindow)->tabs[i].window);
				(*twindow)->tabs[i].window = (WINDOW *) NULL;
			}
			if ((*twindow)->tabs[i].label != (char *) NULL) {
				free((*twindow)->tabs[i].label);
				(*twindow)->tabs[i].label = (char *) NULL;
			}
		}

		free((*twindow)->tabs);
		(*twindow)->tabs = (cdw_tab_t *) NULL;
	}

	free(*twindow);
	*twindow = (cdw_tabs_window_t *) NULL;

	return;
}





void cdw_tabs_window_refresh_current_tab(cdw_tabs_window_t *twindow)
{
	redrawwin(twindow->tabs[twindow->current_tab].window);
	wrefresh(twindow->tabs[twindow->current_tab].window);

	return;
}





void cdw_tabs_window_add_return_key(cdw_tabs_window_t *twindow, int key)
{
	cdw_assert (twindow != (cdw_tabs_window_t *) NULL, "ERROR: \"twindow\" argument is NULL\n");
	cdw_assert (twindow->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the window, can't add another one\n",
		    twindow->n_return_keys, N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key == 0, but 0 is an initializer value\n");

	if (cdw_tabs_window_is_return_key(twindow, key)) {
		cdw_vdm ("WARNING: attempting to add key %d / \"%s\", but it is already on the list of return keys\n",
			 key, cdw_ncurses_key_label(key));
	} else {
		twindow->return_keys[twindow->n_return_keys++] = key;
	}

	return;
}





void cdw_tabs_window_add_return_keys(cdw_tabs_window_t *twindow, ...)
{
	cdw_assert (twindow, "ERROR: twindow is NULL\n");
	cdw_assert (twindow->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the window, can't add another one\n",
		    twindow->n_return_keys, N_RETURN_KEYS_MAX);

	va_list ap;
	va_start(ap, twindow);
	int key = 'a';
	while ((key = va_arg(ap, int)) != 0) {
		cdw_tabs_window_add_return_key(twindow, key);
	}
	va_end(ap);

	return;
}





/**
  \brief Check if given key is one of configured "return" key

  Function checks if given \p key was earlier added to given \p twindow
  as "return control after pressing the key" key.

  \param twindow - tabbed window that you want to query
  \param key - key you want to check if it is added to tabbed window

  \return true if given key was added to given \p twindow
  \return false if given key was not added to given \p twindow
*/
bool cdw_tabs_window_is_return_key(cdw_tabs_window_t *twindow, int key)
{
	cdw_assert (twindow != (cdw_tabs_window_t *) NULL, "ERROR: \"twindow\" argument is NULL\n");
	cdw_assert (key != 0, "ERROR: asking for key = 0, which is an initialization value\n");

	for (int i = 0; i < twindow->n_return_keys; i++) {
		if (twindow->return_keys[i] == key) {
			return true;
		}
	}

	return false;
}





cdw_rv_t cdw_tabs_window_show_tab_by_id(cdw_tabs_window_t *twindow, cdw_id_t id)
{
	for (int i = 0; i < twindow->n_tabs; i++) {
		if (twindow->tabs[i].id == id) {
			twindow->current_tab = i;
			show_panel(twindow->tabs[twindow->current_tab].panel);
			update_panels();
			cdw_main_ui_main_window_wrefresh();
			cdw_tabs_window_refresh_current_tab(twindow);

			return CDW_OK;
		}
	}

	cdw_vdm ("ERROR: failed to switch to tab with id = %lld\n", id);
	return CDW_ERROR;
}
