/*++

Copyright (C) Microsoft Corporation. All Rights Reserved.
SPDX-License-Identifier: BSD-2-Clause-Patent

Module Name:
  AllTestSignatures.h

Abstract:
  This file has test signatures in byte array format for Unit-Testing the
  Enhanced Key Usage (EKU) parsing function.

  EFI_STATUS
  EFIAPI
  VerifyEKUsInPkcs7Signature (
    IN CONST UINT8  *Pkcs7Signature,
    IN CONST UINT32  SignatureSize,
    IN CONST CHAR8  *RequiredEKUs[],
    IN CONST UINT32  RequiredEKUsSize,
    IN BOOLEAN       RequireAllPresent
  );

  Note:  These byte arrays were created by running BinToH.exe on the P7B
         signature files.
--*/

//
// This is the ProductionECCSignature.p7b in byte array format.  It has one
// EKU in it. (Firmware signing)
// "1.3.6.1.4.1.311.76.9.21.1"
//
CONST UINT8  ProductionECCSignature[] =
{
  0x30, 0x82, 0x04, 0xC0, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x80, 0x30, 0x82, 0x03, 0x7C, 0x30, 0x82, 0x03,
  0x22, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x13, 0x33, 0x00, 0x00, 0x00, 0x03, 0x7E, 0x2E, 0x8F,
  0xBD, 0xA5, 0xC9, 0x2C, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x30, 0x0A, 0x06, 0x08, 0x2A,
  0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x81, 0x8F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03,
  0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08,
  0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E,
  0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E,
  0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31, 0x39,
  0x30, 0x37, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x30, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x45, 0x43, 0x43, 0x20, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x20, 0x41,
  0x63, 0x63, 0x65, 0x73, 0x73, 0x6F, 0x72, 0x79, 0x20, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72,
  0x65, 0x20, 0x43, 0x41, 0x20, 0x32, 0x30, 0x31, 0x38, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30,
  0x37, 0x31, 0x32, 0x31, 0x38, 0x33, 0x37, 0x30, 0x37, 0x5A, 0x17, 0x0D, 0x31, 0x39, 0x30, 0x37,
  0x31, 0x32, 0x31, 0x38, 0x33, 0x37, 0x30, 0x37, 0x5A, 0x30, 0x3A, 0x31, 0x38, 0x30, 0x36, 0x06,
  0x03, 0x55, 0x04, 0x03, 0x13, 0x2F, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20,
  0x45, 0x43, 0x43, 0x20, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x20, 0x41, 0x63, 0x63, 0x65,
  0x73, 0x73, 0x6F, 0x72, 0x79, 0x20, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x20, 0x53,
  0x69, 0x67, 0x6E, 0x65, 0x72, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
  0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04,
  0x7D, 0xDF, 0xDE, 0xFA, 0xEE, 0xA5, 0x66, 0x33, 0xAA, 0xCC, 0xA4, 0x18, 0x65, 0x9E, 0x3A, 0xC6,
  0x4B, 0xCB, 0xD5, 0x78, 0x94, 0x14, 0xEE, 0xAA, 0xF3, 0x9D, 0xCB, 0x38, 0x6C, 0x5D, 0xFE, 0x3F,
  0x8A, 0xA9, 0x8D, 0x04, 0xD2, 0xFE, 0xAA, 0x38, 0xF3, 0xD6, 0x4A, 0x7E, 0xCF, 0x12, 0x95, 0xCF,
  0xAD, 0x92, 0x33, 0xF7, 0x38, 0x83, 0xC5, 0xFD, 0xCF, 0xD0, 0x96, 0x08, 0xCA, 0x62, 0x7D, 0x22,
  0xA3, 0x82, 0x01, 0xAF, 0x30, 0x82, 0x01, 0xAB, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01,
  0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x16, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04,
  0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x4C, 0x09, 0x15, 0x01,
  0x30, 0x0C, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x02, 0x30, 0x00, 0x30, 0x1D,
  0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xE7, 0xD1, 0xD4, 0x74, 0x71, 0xF4, 0xC9,
  0x35, 0x04, 0xA7, 0x9A, 0x1E, 0x0C, 0xDD, 0xF4, 0x8D, 0x5C, 0x1E, 0xF2, 0x29, 0x30, 0x37, 0x06,
  0x03, 0x55, 0x1D, 0x11, 0x04, 0x30, 0x30, 0x2E, 0xA4, 0x2C, 0x30, 0x2A, 0x31, 0x10, 0x30, 0x0E,
  0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x07, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x31, 0x16,
  0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x05, 0x13, 0x0D, 0x34, 0x34, 0x30, 0x38, 0x31, 0x31, 0x2B,
  0x34, 0x34, 0x30, 0x38, 0x36, 0x32, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30,
  0x16, 0x80, 0x14, 0x6F, 0x3A, 0xCC, 0x1B, 0x2F, 0x17, 0x5C, 0xED, 0xEB, 0xC8, 0xBF, 0x93, 0xA7,
  0xB5, 0x93, 0xD1, 0x4D, 0x40, 0xAA, 0x03, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1D, 0x1F, 0x04, 0x6E,
  0x30, 0x6C, 0x30, 0x6A, 0xA0, 0x68, 0xA0, 0x66, 0x86, 0x64, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F,
  0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x2E, 0x63,
  0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x6F, 0x70, 0x73, 0x2F, 0x63, 0x72, 0x6C, 0x2F, 0x4D, 0x69,
  0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x25, 0x32, 0x30, 0x45, 0x43, 0x43, 0x25, 0x32, 0x30,
  0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x25, 0x32, 0x30, 0x41, 0x63, 0x63, 0x65, 0x73, 0x73,
  0x6F, 0x72, 0x79, 0x25, 0x32, 0x30, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x25, 0x32,
  0x30, 0x43, 0x41, 0x25, 0x32, 0x30, 0x32, 0x30, 0x31, 0x38, 0x2E, 0x63, 0x72, 0x6C, 0x30, 0x81,
  0x82, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x76, 0x30, 0x74, 0x30,
  0x72, 0x06, 0x08, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x66, 0x68, 0x74, 0x74,
  0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66,
  0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x70, 0x6B, 0x69, 0x6F, 0x70, 0x73, 0x2F, 0x63, 0x65, 0x72,
  0x74, 0x73, 0x2F, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x25, 0x32, 0x30, 0x45,
  0x43, 0x43, 0x25, 0x32, 0x30, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x25, 0x32, 0x30, 0x41,
  0x63, 0x63, 0x65, 0x73, 0x73, 0x6F, 0x72, 0x79, 0x25, 0x32, 0x30, 0x46, 0x69, 0x72, 0x6D, 0x77,
  0x61, 0x72, 0x65, 0x25, 0x32, 0x30, 0x43, 0x41, 0x25, 0x32, 0x30, 0x32, 0x30, 0x31, 0x38, 0x2E,
  0x63, 0x72, 0x74, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03,
  0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x3D, 0xCF, 0xEF, 0xB8, 0xF0, 0x87, 0xEE, 0x7E, 0x80, 0x71,
  0x04, 0x77, 0x68, 0xDC, 0x66, 0x2F, 0x09, 0xA7, 0xF0, 0x76, 0x57, 0x7C, 0x12, 0x5A, 0xE4, 0x80,
  0x5A, 0xA0, 0xF2, 0x2C, 0x45, 0x1B, 0x02, 0x21, 0x00, 0xEB, 0x5F, 0x89, 0xD5, 0xEB, 0x03, 0x46,
  0x9E, 0x08, 0x95, 0x2A, 0x39, 0x30, 0x5D, 0x49, 0xB6, 0x24, 0xCE, 0x78, 0x1B, 0xB8, 0x1D, 0x0F,
  0xD1, 0x24, 0xD0, 0xA9, 0x18, 0x0F, 0x45, 0x4F, 0xF6, 0x31, 0x82, 0x01, 0x17, 0x30, 0x82, 0x01,
  0x13, 0x02, 0x01, 0x01, 0x30, 0x81, 0xA7, 0x30, 0x81, 0x8F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03,
  0x55, 0x04, 0x06, 0x13, 0x02, 0x55, 0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08,
  0x13, 0x0A, 0x57, 0x61, 0x73, 0x68, 0x69, 0x6E, 0x67, 0x74, 0x6F, 0x6E, 0x31, 0x10, 0x30, 0x0E,
  0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x52, 0x65, 0x64, 0x6D, 0x6F, 0x6E, 0x64, 0x31, 0x1E,
  0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31, 0x39,
  0x30, 0x37, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x30, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F,
  0x66, 0x74, 0x20, 0x45, 0x43, 0x43, 0x20, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x20, 0x41,
  0x63, 0x63, 0x65, 0x73, 0x73, 0x6F, 0x72, 0x79, 0x20, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72,
  0x65, 0x20, 0x43, 0x41, 0x20, 0x32, 0x30, 0x31, 0x38, 0x02, 0x13, 0x33, 0x00, 0x00, 0x00, 0x03,
  0x7E, 0x2E, 0x8F, 0xBD, 0xA5, 0xC9, 0x2C, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x30, 0x0D,
  0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06,
  0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x05, 0x00, 0x04, 0x48, 0x30, 0x46, 0x02, 0x21,
  0x00, 0x85, 0xC7, 0x25, 0x5C, 0xF8, 0x5A, 0x13, 0xB7, 0xB6, 0x3A, 0xF1, 0x04, 0x72, 0xAC, 0xB3,
  0xF8, 0xC3, 0xE0, 0xE3, 0xD7, 0x98, 0x1B, 0xCF, 0x04, 0xA8, 0x0C, 0x7B, 0xBB, 0x7D, 0x40, 0xC2,
  0x99, 0x02, 0x21, 0x00, 0xA3, 0x83, 0x12, 0xE4, 0xB6, 0x3C, 0xC9, 0x3F, 0x05, 0xCF, 0x70, 0x57,
  0x54, 0x74, 0x0E, 0xEC, 0x14, 0xAD, 0x2C, 0x66, 0x9D, 0x72, 0x73, 0x62, 0x41, 0xA6, 0xC7, 0x90,
  0x05, 0x35, 0xF5, 0x80,
};

//
// This is the TestSignedWithMultipleCerts.bin.Signature.p7b in byte array
// format.  It has two certs in it, and the leaf has an EKU of
//  "1.3.6.1.4.1.311.76.9.21.1"
//
CONST UINT8  TestSignedMultipleCerts[] =
{
  0x30, 0x82, 0x04, 0x90, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0xC2, 0x30, 0x82, 0x01, 0xC5, 0x30, 0x82, 0x01,
  0x6A, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x16, 0x2F, 0xD8, 0x2B, 0x8A, 0x98, 0x6A, 0x8A,
  0x43, 0xC2, 0x22, 0xA5, 0x47, 0xA1, 0x18, 0xDD, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x26, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D,
  0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D,
  0x31, 0x38, 0x30, 0x37, 0x31, 0x38, 0x32, 0x33, 0x31, 0x36, 0x31, 0x32, 0x5A, 0x17, 0x0D, 0x33,
  0x38, 0x30, 0x37, 0x31, 0x38, 0x32, 0x33, 0x32, 0x36, 0x31, 0x32, 0x5A, 0x30, 0x28, 0x31, 0x26,
  0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1D, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65,
  0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E,
  0x65, 0x72, 0x4C, 0x65, 0x61, 0x66, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00,
  0x04, 0x2E, 0x4C, 0x85, 0xFD, 0x86, 0x85, 0x4C, 0x55, 0x96, 0x17, 0xE3, 0x0C, 0xC1, 0xCB, 0xD9,
  0x95, 0xCF, 0xC7, 0xEA, 0x57, 0x02, 0x5C, 0x4F, 0x92, 0x7C, 0x35, 0xBF, 0x2B, 0x3D, 0xBA, 0x14,
  0xE5, 0x39, 0x9D, 0x9B, 0x64, 0x4A, 0x25, 0x29, 0xD5, 0x61, 0xDA, 0xFD, 0x51, 0x24, 0xB3, 0xE2,
  0x4E, 0x00, 0x59, 0xDF, 0xB4, 0x96, 0x42, 0x59, 0xA4, 0xBC, 0xA7, 0x64, 0x8F, 0x3A, 0x17, 0x33,
  0x55, 0xA3, 0x78, 0x30, 0x76, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04,
  0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x16, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0F, 0x30, 0x0D,
  0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x4C, 0x09, 0x15, 0x01, 0x30, 0x0C, 0x06,
  0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x02, 0x30, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55,
  0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x63, 0x69, 0x95, 0xE3, 0xDD, 0xF4, 0xF8, 0x42,
  0xEA, 0x55, 0x95, 0x54, 0x07, 0xC4, 0xC5, 0x2A, 0xD2, 0xA9, 0x2D, 0x2F, 0x30, 0x1D, 0x06, 0x03,
  0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xD6, 0x4A, 0xF8, 0x77, 0x21, 0x34, 0xEC, 0xC0, 0xAE,
  0x4D, 0x8F, 0x17, 0x86, 0xFF, 0xEA, 0xE3, 0x83, 0x42, 0x32, 0x83, 0x30, 0x0A, 0x06, 0x08, 0x2A,
  0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x9A,
  0xAB, 0xE3, 0x6F, 0xD7, 0xB3, 0x0F, 0x83, 0x8A, 0xA1, 0x64, 0x70, 0xC1, 0xAC, 0x54, 0xE2, 0x59,
  0x7C, 0x8B, 0x11, 0x8F, 0x76, 0xB1, 0x9C, 0x5F, 0xDA, 0x7D, 0x04, 0x3C, 0x73, 0xB5, 0xCF, 0x02,
  0x21, 0x00, 0xB3, 0x31, 0xC1, 0x48, 0x3C, 0xEE, 0xA1, 0x6D, 0xA9, 0xF7, 0xF5, 0x9F, 0xD3, 0x6F,
  0x7D, 0x6F, 0xAB, 0x76, 0x12, 0xEE, 0x01, 0x4D, 0xFC, 0xE6, 0x87, 0xEA, 0xAB, 0x09, 0x98, 0xD8,
  0x97, 0x6F, 0x30, 0x82, 0x01, 0xF5, 0x30, 0x82, 0x01, 0x58, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02,
  0x10, 0x4D, 0x88, 0xBF, 0x04, 0xDF, 0x10, 0x57, 0xBA, 0x43, 0x53, 0x6C, 0xFD, 0xA5, 0x90, 0xA7,
  0x6F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x28, 0x31,
  0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1D, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63,
  0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67,
  0x6E, 0x65, 0x72, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x31, 0x31,
  0x36, 0x32, 0x33, 0x32, 0x31, 0x31, 0x33, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x31, 0x31, 0x36,
  0x32, 0x33, 0x33, 0x31, 0x31, 0x33, 0x5A, 0x30, 0x26, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55,
  0x04, 0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46,
  0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x30,
  0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86,
  0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x0E, 0x58, 0xA3, 0xDB, 0xD3, 0x9C,
  0x21, 0xFC, 0x63, 0x6F, 0xBD, 0xB9, 0xA7, 0x8A, 0xB3, 0xBF, 0x40, 0x87, 0x43, 0x06, 0x53, 0x91,
  0x54, 0x31, 0x11, 0x0D, 0x3B, 0x50, 0x32, 0xEA, 0x11, 0xA3, 0x3D, 0x77, 0x08, 0x9E, 0xBC, 0x06,
  0x5C, 0x21, 0x48, 0x9D, 0x9B, 0x7C, 0x45, 0xFD, 0x4B, 0xD7, 0x0C, 0x04, 0xD4, 0xE0, 0x0C, 0xB7,
  0x6C, 0xBF, 0x22, 0xE1, 0x00, 0x99, 0x75, 0x39, 0x27, 0x59, 0xA3, 0x66, 0x30, 0x64, 0x30, 0x0E,
  0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x12,
  0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xFF, 0x02,
  0x01, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x6D,
  0x8B, 0x24, 0xBF, 0x20, 0x9D, 0x54, 0xAF, 0x04, 0x5B, 0x84, 0x91, 0x96, 0xAE, 0x99, 0xC1, 0x23,
  0x1C, 0x0E, 0xAC, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x63, 0x69,
  0x95, 0xE3, 0xDD, 0xF4, 0xF8, 0x42, 0xEA, 0x55, 0x95, 0x54, 0x07, 0xC4, 0xC5, 0x2A, 0xD2, 0xA9,
  0x2D, 0x2F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03, 0x81,
  0x8A, 0x00, 0x30, 0x81, 0x86, 0x02, 0x41, 0x2A, 0xD4, 0xFC, 0x35, 0x28, 0x73, 0x9F, 0xB3, 0x8A,
  0xE3, 0x13, 0x7F, 0xCA, 0x36, 0x88, 0x1E, 0xD0, 0x95, 0xDC, 0xE9, 0x8A, 0x9D, 0xC0, 0x27, 0x8C,
  0xD8, 0x57, 0x57, 0x5F, 0xDC, 0x13, 0x53, 0x68, 0xED, 0x2F, 0x4D, 0x6A, 0x77, 0xA2, 0x55, 0x48,
  0xCB, 0x24, 0x55, 0x3A, 0x30, 0x32, 0xE5, 0x5C, 0xDF, 0x96, 0xF1, 0x15, 0xD4, 0xE7, 0xCD, 0x6A,
  0x26, 0xBD, 0x7B, 0x92, 0x50, 0x74, 0xE4, 0x56, 0x02, 0x41, 0x46, 0x33, 0x61, 0x30, 0xFF, 0xA8,
  0x53, 0x31, 0x38, 0xCA, 0x6B, 0x77, 0x88, 0x36, 0x9A, 0x0B, 0x57, 0xC3, 0xD1, 0xB2, 0xA7, 0x40,
  0x53, 0x6A, 0xD1, 0x6C, 0x8A, 0x8B, 0x3C, 0x29, 0xC1, 0x2B, 0x82, 0x61, 0x2F, 0x1A, 0x9F, 0x2C,
  0x9A, 0x81, 0xD0, 0x2B, 0x3E, 0x0D, 0x99, 0xB7, 0xD4, 0x7E, 0xA7, 0xEA, 0x0C, 0x8E, 0x13, 0x0E,
  0x53, 0x65, 0x48, 0x17, 0xE5, 0xFA, 0x91, 0x79, 0x98, 0xC7, 0xA1, 0x31, 0x81, 0xA6, 0x30, 0x81,
  0xA3, 0x02, 0x01, 0x01, 0x30, 0x3A, 0x30, 0x26, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69,
  0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x02, 0x10,
  0x16, 0x2F, 0xD8, 0x2B, 0x8A, 0x98, 0x6A, 0x8A, 0x43, 0xC2, 0x22, 0xA5, 0x47, 0xA1, 0x18, 0xDD,
  0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30,
  0x0B, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x05, 0x00, 0x04, 0x46, 0x30, 0x44,
  0x02, 0x20, 0x40, 0xD4, 0xD4, 0xAF, 0x89, 0xFA, 0x83, 0x44, 0x7B, 0x59, 0xB4, 0x2B, 0x99, 0xF0,
  0xD4, 0xB0, 0x75, 0x2E, 0x56, 0xAA, 0x0A, 0xDA, 0xCC, 0x82, 0xD5, 0xBC, 0x8D, 0xC4, 0x00, 0xF6,
  0x46, 0xA4, 0x02, 0x20, 0x6F, 0xA7, 0xE1, 0x32, 0xB2, 0xF2, 0x5A, 0x69, 0xC5, 0x45, 0xB6, 0x35,
  0x04, 0x05, 0x9B, 0xAC, 0x44, 0xB4, 0xA0, 0xE1, 0x25, 0x92, 0xE8, 0x3E, 0x8F, 0x85, 0xA3, 0xBE,
  0x34, 0xFE, 0x78, 0x31,
};

//
// This is the TestSignedWithMultipleEKUsInCert.bin.Signature.p7b signature in
// byte array format.  It has two EKU's in it:
// "1.3.6.1.4.1.311.76.9.21.1"
// "1.3.6.1.4.1.311.76.9.21.2"
//
CONST UINT8  TestSignedWithMultipleEKUsInCert[] =
{
  0x30, 0x82, 0x06, 0x2C, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x04, 0x9E, 0x30, 0x82, 0x01, 0xF5, 0x30, 0x82, 0x01,
  0x58, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x4D, 0x88, 0xBF, 0x04, 0xDF, 0x10, 0x57, 0xBA,
  0x43, 0x53, 0x6C, 0xFD, 0xA5, 0x90, 0xA7, 0x6F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x1D, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D,
  0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x1E,
  0x17, 0x0D, 0x31, 0x38, 0x30, 0x31, 0x31, 0x36, 0x32, 0x33, 0x32, 0x31, 0x31, 0x33, 0x5A, 0x17,
  0x0D, 0x33, 0x38, 0x30, 0x31, 0x31, 0x36, 0x32, 0x33, 0x33, 0x31, 0x31, 0x33, 0x5A, 0x30, 0x26,
  0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61,
  0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69,
  0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00,
  0x04, 0x0E, 0x58, 0xA3, 0xDB, 0xD3, 0x9C, 0x21, 0xFC, 0x63, 0x6F, 0xBD, 0xB9, 0xA7, 0x8A, 0xB3,
  0xBF, 0x40, 0x87, 0x43, 0x06, 0x53, 0x91, 0x54, 0x31, 0x11, 0x0D, 0x3B, 0x50, 0x32, 0xEA, 0x11,
  0xA3, 0x3D, 0x77, 0x08, 0x9E, 0xBC, 0x06, 0x5C, 0x21, 0x48, 0x9D, 0x9B, 0x7C, 0x45, 0xFD, 0x4B,
  0xD7, 0x0C, 0x04, 0xD4, 0xE0, 0x0C, 0xB7, 0x6C, 0xBF, 0x22, 0xE1, 0x00, 0x99, 0x75, 0x39, 0x27,
  0x59, 0xA3, 0x66, 0x30, 0x64, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04,
  0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04,
  0x08, 0x30, 0x06, 0x01, 0x01, 0xFF, 0x02, 0x01, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23,
  0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x6D, 0x8B, 0x24, 0xBF, 0x20, 0x9D, 0x54, 0xAF, 0x04, 0x5B,
  0x84, 0x91, 0x96, 0xAE, 0x99, 0xC1, 0x23, 0x1C, 0x0E, 0xAC, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D,
  0x0E, 0x04, 0x16, 0x04, 0x14, 0x63, 0x69, 0x95, 0xE3, 0xDD, 0xF4, 0xF8, 0x42, 0xEA, 0x55, 0x95,
  0x54, 0x07, 0xC4, 0xC5, 0x2A, 0xD2, 0xA9, 0x2D, 0x2F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48,
  0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03, 0x81, 0x8A, 0x00, 0x30, 0x81, 0x86, 0x02, 0x41, 0x2A, 0xD4,
  0xFC, 0x35, 0x28, 0x73, 0x9F, 0xB3, 0x8A, 0xE3, 0x13, 0x7F, 0xCA, 0x36, 0x88, 0x1E, 0xD0, 0x95,
  0xDC, 0xE9, 0x8A, 0x9D, 0xC0, 0x27, 0x8C, 0xD8, 0x57, 0x57, 0x5F, 0xDC, 0x13, 0x53, 0x68, 0xED,
  0x2F, 0x4D, 0x6A, 0x77, 0xA2, 0x55, 0x48, 0xCB, 0x24, 0x55, 0x3A, 0x30, 0x32, 0xE5, 0x5C, 0xDF,
  0x96, 0xF1, 0x15, 0xD4, 0xE7, 0xCD, 0x6A, 0x26, 0xBD, 0x7B, 0x92, 0x50, 0x74, 0xE4, 0x56, 0x02,
  0x41, 0x46, 0x33, 0x61, 0x30, 0xFF, 0xA8, 0x53, 0x31, 0x38, 0xCA, 0x6B, 0x77, 0x88, 0x36, 0x9A,
  0x0B, 0x57, 0xC3, 0xD1, 0xB2, 0xA7, 0x40, 0x53, 0x6A, 0xD1, 0x6C, 0x8A, 0x8B, 0x3C, 0x29, 0xC1,
  0x2B, 0x82, 0x61, 0x2F, 0x1A, 0x9F, 0x2C, 0x9A, 0x81, 0xD0, 0x2B, 0x3E, 0x0D, 0x99, 0xB7, 0xD4,
  0x7E, 0xA7, 0xEA, 0x0C, 0x8E, 0x13, 0x0E, 0x53, 0x65, 0x48, 0x17, 0xE5, 0xFA, 0x91, 0x79, 0x98,
  0xC7, 0xA1, 0x30, 0x82, 0x02, 0xA1, 0x30, 0x82, 0x02, 0x47, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02,
  0x10, 0x29, 0xBB, 0x52, 0x25, 0xED, 0x34, 0xB7, 0xBA, 0x40, 0xF8, 0xC3, 0x14, 0xB5, 0xC8, 0x86,
  0x9E, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x26, 0x31,
  0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63,
  0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67,
  0x6E, 0x65, 0x72, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x37, 0x32, 0x37, 0x32,
  0x30, 0x33, 0x34, 0x31, 0x31, 0x5A, 0x17, 0x0D, 0x32, 0x33, 0x30, 0x37, 0x32, 0x37, 0x32, 0x30,
  0x34, 0x34, 0x31, 0x31, 0x5A, 0x30, 0x3A, 0x31, 0x38, 0x30, 0x36, 0x06, 0x03, 0x55, 0x04, 0x03,
  0x0C, 0x2F, 0x54, 0x65, 0x73, 0x74, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x46, 0x69, 0x72,
  0x6D, 0x77, 0x61, 0x72, 0x65, 0x4C, 0x65, 0x61, 0x66, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x4D,
  0x75, 0x6C, 0x74, 0x69, 0x70, 0x6C, 0x65, 0x45, 0x4B, 0x55, 0x73, 0x49, 0x6E, 0x43, 0x65, 0x72,
  0x74, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01,
  0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01,
  0x01, 0x00, 0xD0, 0x61, 0xA5, 0xCD, 0x49, 0x4B, 0x5C, 0x88, 0x89, 0x09, 0xCA, 0xE7, 0xEB, 0x58,
  0x91, 0x8F, 0x59, 0xBE, 0xE8, 0x16, 0xC0, 0xEF, 0x75, 0xE5, 0x18, 0x88, 0xE8, 0xCE, 0xFA, 0xD2,
  0xB0, 0x77, 0xC1, 0xD2, 0xF2, 0x9E, 0x65, 0xE8, 0xDB, 0x6B, 0x8C, 0x93, 0xE5, 0x39, 0x1D, 0xF9,
  0x37, 0x2A, 0x52, 0x7F, 0xC3, 0x93, 0xD7, 0x9F, 0x03, 0x83, 0xBC, 0x6E, 0xC6, 0x83, 0x64, 0x15,
  0x1A, 0x5C, 0x74, 0x41, 0x40, 0x21, 0xC9, 0x25, 0xF0, 0x14, 0xA3, 0x29, 0x97, 0x29, 0x8C, 0x5D,
  0xED, 0xE9, 0x8D, 0x66, 0xCC, 0xD1, 0xD1, 0xE0, 0x08, 0x52, 0xA5, 0x93, 0x07, 0xCB, 0xF8, 0x95,
  0x2C, 0x8D, 0x29, 0x84, 0x10, 0x5F, 0x57, 0x52, 0x10, 0x75, 0x35, 0x8A, 0x84, 0xE3, 0xAE, 0x10,
  0x57, 0x29, 0xE9, 0xEB, 0x6E, 0xD2, 0xC7, 0x6D, 0x35, 0x43, 0x67, 0x82, 0xE3, 0x98, 0x29, 0x9C,
  0x59, 0x77, 0xA0, 0x00, 0x27, 0xB2, 0x5D, 0xB9, 0x84, 0x31, 0x89, 0x2F, 0x77, 0xC5, 0xB9, 0x23,
  0xE0, 0x14, 0x94, 0x6F, 0x33, 0xD5, 0x3B, 0x58, 0xFB, 0xD6, 0xB6, 0xE5, 0xD4, 0x09, 0x3D, 0xB2,
  0xE7, 0x87, 0xC6, 0xEA, 0x73, 0x61, 0x6D, 0xDA, 0xFA, 0x95, 0x2B, 0x2B, 0x8A, 0xEC, 0x7E, 0x68,
  0x0F, 0x98, 0x16, 0x08, 0xFF, 0xB2, 0xC1, 0xBF, 0xAB, 0x3D, 0xB0, 0xC2, 0xD5, 0x16, 0xAE, 0x71,
  0x54, 0x48, 0xD1, 0x52, 0x72, 0xFD, 0x42, 0x09, 0x66, 0xAF, 0x50, 0xDE, 0x4C, 0x3C, 0x6A, 0xBC,
  0x1F, 0xC5, 0x0A, 0xA0, 0xC5, 0x9A, 0xF1, 0xB2, 0xB2, 0x3F, 0x14, 0xE1, 0x93, 0x07, 0xF0, 0x00,
  0x04, 0xEF, 0x16, 0x76, 0x18, 0x10, 0x65, 0x63, 0x42, 0x42, 0x12, 0xD0, 0x65, 0xF5, 0xA2, 0xCF,
  0x90, 0xBE, 0x65, 0x57, 0x25, 0x8C, 0xAC, 0x5E, 0x05, 0x31, 0x91, 0x9E, 0x4D, 0x18, 0xBC, 0x15,
  0x30, 0x7B, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x78, 0x30, 0x76, 0x30, 0x0E, 0x06, 0x03, 0x55,
  0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x01, 0x86, 0x30, 0x24, 0x06, 0x03, 0x55,
  0x1D, 0x25, 0x04, 0x1D, 0x30, 0x1B, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x4C,
  0x09, 0x15, 0x01, 0x06, 0x0C, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x4C, 0x09, 0x15, 0x01,
  0x02, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x63, 0x69,
  0x95, 0xE3, 0xDD, 0xF4, 0xF8, 0x42, 0xEA, 0x55, 0x95, 0x54, 0x07, 0xC4, 0xC5, 0x2A, 0xD2, 0xA9,
  0x2D, 0x2F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0xC0, 0xDC, 0xCB,
  0x76, 0x52, 0x84, 0x91, 0xF5, 0x37, 0x0D, 0x85, 0xB6, 0x45, 0x23, 0xFA, 0xA0, 0xAD, 0xC1, 0x1D,
  0x92, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00,
  0x30, 0x45, 0x02, 0x21, 0x00, 0xF4, 0xB0, 0x8A, 0x91, 0x96, 0x61, 0xDF, 0xC3, 0x54, 0xA5, 0xB3,
  0x1A, 0x3C, 0x28, 0x06, 0xC0, 0xA1, 0xF5, 0x72, 0xD1, 0x02, 0x5C, 0x69, 0xB6, 0x6C, 0xA8, 0xDC,
  0x6B, 0x78, 0xD9, 0x0C, 0x9A, 0x02, 0x20, 0x37, 0x2C, 0x61, 0x9E, 0x6E, 0x5F, 0xC6, 0xFA, 0x92,
  0x52, 0xB0, 0x4E, 0x52, 0xC7, 0xF1, 0xE9, 0x0F, 0xB6, 0xD5, 0x08, 0x10, 0x41, 0x3A, 0x61, 0xFD,
  0xC5, 0x70, 0x87, 0xBB, 0x49, 0x82, 0xEA, 0x31, 0x82, 0x01, 0x65, 0x30, 0x82, 0x01, 0x61, 0x02,
  0x01, 0x01, 0x30, 0x3A, 0x30, 0x26, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D,
  0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x02, 0x10, 0x29, 0xBB,
  0x52, 0x25, 0xED, 0x34, 0xB7, 0xBA, 0x40, 0xF8, 0xC3, 0x14, 0xB5, 0xC8, 0x86, 0x9E, 0x30, 0x0D,
  0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0D, 0x06,
  0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82, 0x01, 0x00,
  0xBB, 0x51, 0xDD, 0x8F, 0xB4, 0x99, 0x37, 0x13, 0x94, 0x6F, 0x09, 0x05, 0xBA, 0x5B, 0x8D, 0x86,
  0x9A, 0xB0, 0x51, 0xEB, 0x6B, 0x10, 0xD7, 0xD1, 0x95, 0x4A, 0x4D, 0x23, 0xA3, 0x37, 0x82, 0xDF,
  0x1F, 0x24, 0xE4, 0x62, 0x4F, 0xD4, 0x52, 0x73, 0xF8, 0x39, 0x2B, 0x46, 0xF4, 0x61, 0xCE, 0xF1,
  0x64, 0xD7, 0x60, 0xC9, 0xCC, 0xCA, 0x8D, 0x94, 0xA5, 0x67, 0x68, 0x2B, 0xE4, 0xB0, 0xD7, 0x64,
  0xC8, 0xF3, 0x9C, 0x98, 0x2C, 0x0A, 0x69, 0x56, 0xD6, 0x59, 0xDF, 0x4D, 0x38, 0xE7, 0xD4, 0x63,
  0x2B, 0x74, 0xA1, 0xA3, 0xF9, 0xE1, 0x1A, 0x12, 0x2B, 0x04, 0x46, 0x11, 0xA8, 0x26, 0x00, 0x15,
  0x4F, 0x14, 0x7F, 0xB1, 0xB4, 0x25, 0x0C, 0x87, 0xB0, 0x56, 0x8C, 0xBE, 0x0D, 0xFA, 0x66, 0x6F,
  0x8A, 0xCB, 0x0A, 0x88, 0x3B, 0xF6, 0x52, 0xCD, 0xBB, 0xA2, 0xA9, 0xAD, 0xAD, 0x09, 0x0D, 0xDA,
  0x79, 0x8A, 0xAE, 0xE5, 0x9D, 0xCE, 0x5C, 0x60, 0x99, 0x67, 0x26, 0xC4, 0xAF, 0xD4, 0x01, 0xB1,
  0x6D, 0xD1, 0x8C, 0x4D, 0x1D, 0xBD, 0xAB, 0x42, 0x2C, 0xF2, 0xD8, 0x07, 0x3F, 0xDC, 0x5E, 0x68,
  0x4C, 0xD9, 0xDB, 0xC0, 0x93, 0xBB, 0x8E, 0xA0, 0xA2, 0x09, 0x5F, 0x03, 0x58, 0x48, 0x54, 0xF6,
  0xF0, 0xDA, 0x7A, 0x7F, 0xC4, 0xD2, 0x9F, 0x64, 0xF5, 0x85, 0x86, 0x48, 0xDF, 0x7F, 0x86, 0x14,
  0x7C, 0x62, 0x77, 0x1D, 0x22, 0xE5, 0x34, 0x42, 0xE4, 0x49, 0x38, 0xB4, 0xE4, 0x22, 0xFB, 0x5F,
  0xB5, 0xEA, 0xBA, 0x90, 0x51, 0xC0, 0xC6, 0x94, 0x09, 0xE4, 0xB7, 0x15, 0x3F, 0x07, 0x23, 0xE8,
  0x46, 0x93, 0xA5, 0x7B, 0x7A, 0x91, 0xDA, 0x8E, 0x7C, 0xAF, 0xBD, 0x41, 0xB9, 0xDE, 0x85, 0x04,
  0xBC, 0x08, 0x6C, 0x08, 0x56, 0x16, 0xDB, 0xB5, 0xEE, 0x65, 0x76, 0xE9, 0x78, 0xD3, 0xDD, 0xD8,
};

//
// This is the TestSignedWithProductId10001.bin.p7b in byte array format.
// It has two EKU's in it:  (Product ID is 10001)
// "1.3.6.1.4.1.311.76.9.21.1"
// "1.3.6.1.4.1.311.76.9.21.1.10001"
//
CONST UINT8  TestSignedWithProductId10001[] =
{
  0x30, 0x82, 0x06, 0x1E, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x04, 0x90, 0x30, 0x82, 0x01, 0xF5, 0x30, 0x82, 0x01,
  0x58, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x4D, 0x88, 0xBF, 0x04, 0xDF, 0x10, 0x57, 0xBA,
  0x43, 0x53, 0x6C, 0xFD, 0xA5, 0x90, 0xA7, 0x6F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x1D, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D,
  0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x1E,
  0x17, 0x0D, 0x31, 0x38, 0x30, 0x31, 0x31, 0x36, 0x32, 0x33, 0x32, 0x31, 0x31, 0x33, 0x5A, 0x17,
  0x0D, 0x33, 0x38, 0x30, 0x31, 0x31, 0x36, 0x32, 0x33, 0x33, 0x31, 0x31, 0x33, 0x5A, 0x30, 0x26,
  0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61,
  0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69,
  0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00,
  0x04, 0x0E, 0x58, 0xA3, 0xDB, 0xD3, 0x9C, 0x21, 0xFC, 0x63, 0x6F, 0xBD, 0xB9, 0xA7, 0x8A, 0xB3,
  0xBF, 0x40, 0x87, 0x43, 0x06, 0x53, 0x91, 0x54, 0x31, 0x11, 0x0D, 0x3B, 0x50, 0x32, 0xEA, 0x11,
  0xA3, 0x3D, 0x77, 0x08, 0x9E, 0xBC, 0x06, 0x5C, 0x21, 0x48, 0x9D, 0x9B, 0x7C, 0x45, 0xFD, 0x4B,
  0xD7, 0x0C, 0x04, 0xD4, 0xE0, 0x0C, 0xB7, 0x6C, 0xBF, 0x22, 0xE1, 0x00, 0x99, 0x75, 0x39, 0x27,
  0x59, 0xA3, 0x66, 0x30, 0x64, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04,
  0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04,
  0x08, 0x30, 0x06, 0x01, 0x01, 0xFF, 0x02, 0x01, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23,
  0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x6D, 0x8B, 0x24, 0xBF, 0x20, 0x9D, 0x54, 0xAF, 0x04, 0x5B,
  0x84, 0x91, 0x96, 0xAE, 0x99, 0xC1, 0x23, 0x1C, 0x0E, 0xAC, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D,
  0x0E, 0x04, 0x16, 0x04, 0x14, 0x63, 0x69, 0x95, 0xE3, 0xDD, 0xF4, 0xF8, 0x42, 0xEA, 0x55, 0x95,
  0x54, 0x07, 0xC4, 0xC5, 0x2A, 0xD2, 0xA9, 0x2D, 0x2F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48,
  0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03, 0x81, 0x8A, 0x00, 0x30, 0x81, 0x86, 0x02, 0x41, 0x2A, 0xD4,
  0xFC, 0x35, 0x28, 0x73, 0x9F, 0xB3, 0x8A, 0xE3, 0x13, 0x7F, 0xCA, 0x36, 0x88, 0x1E, 0xD0, 0x95,
  0xDC, 0xE9, 0x8A, 0x9D, 0xC0, 0x27, 0x8C, 0xD8, 0x57, 0x57, 0x5F, 0xDC, 0x13, 0x53, 0x68, 0xED,
  0x2F, 0x4D, 0x6A, 0x77, 0xA2, 0x55, 0x48, 0xCB, 0x24, 0x55, 0x3A, 0x30, 0x32, 0xE5, 0x5C, 0xDF,
  0x96, 0xF1, 0x15, 0xD4, 0xE7, 0xCD, 0x6A, 0x26, 0xBD, 0x7B, 0x92, 0x50, 0x74, 0xE4, 0x56, 0x02,
  0x41, 0x46, 0x33, 0x61, 0x30, 0xFF, 0xA8, 0x53, 0x31, 0x38, 0xCA, 0x6B, 0x77, 0x88, 0x36, 0x9A,
  0x0B, 0x57, 0xC3, 0xD1, 0xB2, 0xA7, 0x40, 0x53, 0x6A, 0xD1, 0x6C, 0x8A, 0x8B, 0x3C, 0x29, 0xC1,
  0x2B, 0x82, 0x61, 0x2F, 0x1A, 0x9F, 0x2C, 0x9A, 0x81, 0xD0, 0x2B, 0x3E, 0x0D, 0x99, 0xB7, 0xD4,
  0x7E, 0xA7, 0xEA, 0x0C, 0x8E, 0x13, 0x0E, 0x53, 0x65, 0x48, 0x17, 0xE5, 0xFA, 0x91, 0x79, 0x98,
  0xC7, 0xA1, 0x30, 0x82, 0x02, 0x93, 0x30, 0x82, 0x02, 0x39, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02,
  0x10, 0x1F, 0x65, 0xF1, 0x29, 0x3B, 0x89, 0x6A, 0xBF, 0x43, 0xAF, 0xC7, 0x30, 0x36, 0xD1, 0x0D,
  0x2B, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x26, 0x31,
  0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63,
  0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67,
  0x6E, 0x65, 0x72, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38, 0x30, 0x32, 0x31,
  0x36, 0x34, 0x32, 0x32, 0x33, 0x5A, 0x17, 0x0D, 0x32, 0x33, 0x30, 0x38, 0x30, 0x32, 0x31, 0x36,
  0x35, 0x32, 0x32, 0x33, 0x5A, 0x30, 0x2B, 0x31, 0x29, 0x30, 0x27, 0x06, 0x03, 0x55, 0x04, 0x03,
  0x0C, 0x20, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69, 0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69,
  0x67, 0x6E, 0x65, 0x72, 0x50, 0x72, 0x6F, 0x64, 0x75, 0x63, 0x74, 0x49, 0x44, 0x31, 0x30, 0x30,
  0x30, 0x31, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
  0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82,
  0x01, 0x01, 0x00, 0x8C, 0x26, 0x35, 0xBC, 0x18, 0x06, 0x44, 0x65, 0xF6, 0x55, 0x29, 0x78, 0x26,
  0xCA, 0x57, 0xD0, 0x4C, 0x31, 0x84, 0x5B, 0x64, 0xDE, 0x8E, 0x34, 0x84, 0x45, 0x12, 0xE8, 0x5A,
  0x86, 0x4B, 0x5A, 0x61, 0x4C, 0xD8, 0xE2, 0x7D, 0x3B, 0xBD, 0x25, 0x58, 0x30, 0x1D, 0x5E, 0x65,
  0x50, 0xC3, 0x07, 0xF2, 0x50, 0xE1, 0x48, 0x2A, 0xD9, 0x70, 0x5F, 0xF3, 0x2B, 0xF1, 0x1F, 0x21,
  0x06, 0x62, 0x81, 0xE9, 0x8E, 0xDA, 0xDC, 0x69, 0xA3, 0x6E, 0x2F, 0xD2, 0x3A, 0x79, 0x01, 0xAF,
  0xA1, 0x2A, 0x28, 0xDD, 0x32, 0x4E, 0xCE, 0x40, 0xB8, 0x41, 0xA4, 0x7D, 0x3D, 0x70, 0x3F, 0xC4,
  0xDA, 0x61, 0x18, 0xBF, 0xFC, 0xAB, 0x16, 0xE3, 0x65, 0x3F, 0x6D, 0x76, 0x37, 0xAD, 0x1F, 0x11,
  0x05, 0x0F, 0xC2, 0x56, 0xD1, 0x45, 0xBD, 0x25, 0xCC, 0x19, 0xC4, 0x2A, 0x03, 0xED, 0x3B, 0x05,
  0xA7, 0x87, 0x30, 0xE6, 0x84, 0x11, 0x88, 0xD0, 0x91, 0x4E, 0x52, 0xB5, 0x97, 0xDF, 0x57, 0xEF,
  0x5E, 0x72, 0x68, 0x49, 0x85, 0xC8, 0x7F, 0xA7, 0x5F, 0x7B, 0xE7, 0xCE, 0xA1, 0x53, 0xBA, 0xF8,
  0xA9, 0xDF, 0x8C, 0x8C, 0x72, 0xD2, 0xA9, 0x7A, 0xF5, 0x07, 0x4A, 0x75, 0x1A, 0x0F, 0x50, 0x9E,
  0xEA, 0x4C, 0xFC, 0xC7, 0x74, 0xE2, 0x4D, 0xD1, 0x01, 0xC4, 0xB6, 0xCA, 0xA5, 0xCC, 0x23, 0xFA,
  0xDE, 0xF2, 0x9C, 0x8B, 0xA6, 0x32, 0x8D, 0xED, 0x97, 0xCC, 0x50, 0xB0, 0xB6, 0x1C, 0x89, 0x64,
  0xFC, 0xD7, 0xD0, 0x19, 0x01, 0x50, 0xD9, 0xBC, 0x78, 0xB1, 0x76, 0x6D, 0x01, 0xF8, 0xEC, 0x68,
  0x97, 0x20, 0x32, 0xD0, 0x57, 0x76, 0x93, 0x73, 0xC9, 0x2B, 0x68, 0x9D, 0x5E, 0xE1, 0x70, 0xDD,
  0x6F, 0x3A, 0x09, 0x20, 0x63, 0xE9, 0xD9, 0x30, 0x05, 0x14, 0x4A, 0x76, 0xA7, 0x3E, 0x61, 0xCD,
  0x2F, 0x6D, 0xE3, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x79, 0x30, 0x77, 0x30, 0x0E, 0x06, 0x03,
  0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x01, 0x86, 0x30, 0x25, 0x06, 0x03,
  0x55, 0x1D, 0x25, 0x04, 0x1E, 0x30, 0x1C, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x4C, 0x09, 0x15, 0x01, 0x06, 0x0D, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x4C, 0x09, 0x15,
  0x01, 0xCE, 0x11, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14,
  0x63, 0x69, 0x95, 0xE3, 0xDD, 0xF4, 0xF8, 0x42, 0xEA, 0x55, 0x95, 0x54, 0x07, 0xC4, 0xC5, 0x2A,
  0xD2, 0xA9, 0x2D, 0x2F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x09,
  0x10, 0xEE, 0x7B, 0xB0, 0x56, 0xB0, 0xCE, 0x42, 0x83, 0x84, 0x90, 0x70, 0xAF, 0x8D, 0x58, 0x1B,
  0x77, 0x43, 0x0E, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03,
  0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xEE, 0x2A, 0x00, 0x7E, 0xC6, 0x4E, 0x6F, 0x22, 0xCC,
  0x16, 0xEA, 0xAC, 0x90, 0x3D, 0xD7, 0x6E, 0xAA, 0xBF, 0x7E, 0xC7, 0x45, 0x3F, 0x36, 0x22, 0x81,
  0xC6, 0x38, 0xD3, 0x1E, 0xDF, 0x5F, 0x2C, 0x02, 0x20, 0x21, 0xEC, 0xCF, 0x78, 0x1D, 0xDB, 0x4F,
  0x50, 0xE8, 0xD6, 0x07, 0xB0, 0x6A, 0xE9, 0x1E, 0xBF, 0x9C, 0xE8, 0xB9, 0xF7, 0xC6, 0xEA, 0x04,
  0x7B, 0x5E, 0xD8, 0x67, 0xAB, 0xD2, 0xDE, 0x77, 0xB9, 0x31, 0x82, 0x01, 0x65, 0x30, 0x82, 0x01,
  0x61, 0x02, 0x01, 0x01, 0x30, 0x3A, 0x30, 0x26, 0x31, 0x24, 0x30, 0x22, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x0C, 0x1B, 0x53, 0x75, 0x72, 0x66, 0x61, 0x63, 0x65, 0x54, 0x65, 0x73, 0x74, 0x46, 0x69,
  0x72, 0x6D, 0x77, 0x61, 0x72, 0x65, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x43, 0x41, 0x02, 0x10,
  0x1F, 0x65, 0xF1, 0x29, 0x3B, 0x89, 0x6A, 0xBF, 0x43, 0xAF, 0xC7, 0x30, 0x36, 0xD1, 0x0D, 0x2B,
  0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30,
  0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
  0x01, 0x00, 0x1F, 0x7F, 0x30, 0xDC, 0xD4, 0x12, 0x9D, 0x0B, 0x71, 0x30, 0x69, 0x91, 0xCC, 0x8A,
  0xFB, 0xF6, 0x54, 0x14, 0x4C, 0x0B, 0x78, 0xCE, 0xD3, 0xC6, 0xCE, 0x3E, 0xAC, 0xCC, 0x0F, 0xCE,
  0x3A, 0xFC, 0x63, 0x96, 0xD2, 0x7B, 0xDB, 0xF7, 0xE7, 0x6D, 0x64, 0x52, 0x5C, 0x5D, 0x19, 0xBF,
  0xE3, 0x04, 0x71, 0x8A, 0xD1, 0x2A, 0xB6, 0x5A, 0x8C, 0xFC, 0xE2, 0x51, 0x27, 0x64, 0x9B, 0xA8,
  0x84, 0x69, 0x60, 0x4E, 0x56, 0x83, 0x91, 0x4F, 0xEB, 0x5A, 0xB7, 0x73, 0xE1, 0xAC, 0x24, 0xD2,
  0x60, 0xD8, 0x58, 0x21, 0x51, 0x7B, 0x87, 0x93, 0x5B, 0x5F, 0x91, 0x4B, 0x8C, 0xFE, 0xA5, 0x16,
  0x2E, 0x33, 0x4D, 0xBB, 0x0C, 0x95, 0x5C, 0xC3, 0x9C, 0x3D, 0xBB, 0x58, 0x17, 0x39, 0x1E, 0x10,
  0x47, 0xCD, 0xEE, 0xD3, 0xE7, 0x73, 0x44, 0x7D, 0x35, 0xD9, 0x9F, 0xC3, 0xAC, 0x11, 0x2E, 0x29,
  0x54, 0x80, 0x8F, 0x19, 0xDC, 0xC3, 0x11, 0x3D, 0x75, 0xF0, 0x3B, 0x34, 0x63, 0xB1, 0x59, 0x1D,
  0x4B, 0x66, 0xBE, 0xB8, 0xAC, 0x36, 0x71, 0x63, 0xD4, 0x41, 0xCF, 0x24, 0x2A, 0x3F, 0xC5, 0x25,
  0xD2, 0x45, 0x9B, 0xD3, 0x77, 0xD8, 0xD8, 0x1A, 0xC6, 0xAE, 0xE3, 0xF5, 0xC5, 0xEB, 0xF3, 0x92,
  0x27, 0x3C, 0x3F, 0x48, 0x96, 0xF2, 0xB4, 0x66, 0x0A, 0xFB, 0xBF, 0x76, 0xDC, 0x1E, 0xC5, 0xC6,
  0x4D, 0xA1, 0x5C, 0x13, 0x22, 0xF8, 0x89, 0xE2, 0x15, 0x43, 0x84, 0xEF, 0xC8, 0x30, 0xE5, 0xF0,
  0xBA, 0x51, 0x96, 0x3E, 0xF9, 0xCB, 0x41, 0xE6, 0x36, 0xD4, 0xE3, 0xDA, 0x71, 0xC8, 0x5E, 0x43,
  0x7A, 0xBE, 0x8E, 0xC0, 0xA5, 0xEF, 0x66, 0xFD, 0x63, 0xD3, 0xF1, 0xBF, 0x66, 0x4F, 0x35, 0x24,
  0x66, 0xC9, 0xCD, 0x1E, 0x3D, 0x71, 0x19, 0x84, 0x1A, 0x9D, 0xA5, 0x56, 0xF4, 0xF9, 0xE0, 0x88,
  0x7C, 0x52,
};

//
// This is TestSignEKUsWith1CertInSignature.bin.p7b in byte array format.
// It has one EKU in it:
// "1.3.6.1.4.1.311.76.9.21.1"
//
CONST UINT8  TestSignEKUsWith1CertInSignature[] =
{
  0x30, 0x82, 0x02, 0x7D, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x01, 0xB2, 0x30, 0x82, 0x01, 0xAE, 0x30, 0x82, 0x01,
  0x53, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x3E, 0xEE, 0x4B, 0xFB, 0x00, 0x35, 0xC8, 0x96,
  0x41, 0xF4, 0xF6, 0x57, 0x16, 0xAD, 0x20, 0xCB, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x49,
  0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38,
  0x31, 0x35, 0x31, 0x37, 0x33, 0x35, 0x35, 0x30, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38, 0x31,
  0x35, 0x31, 0x37, 0x34, 0x35, 0x35, 0x30, 0x5A, 0x30, 0x23, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03,
  0x55, 0x04, 0x03, 0x0C, 0x18, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73,
  0x69, 0x6E, 0x67, 0x4C, 0x65, 0x61, 0x66, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x30, 0x59, 0x30,
  0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xF6, 0xEA, 0x5E, 0x14, 0x20, 0x6D, 0x11, 0xC0,
  0xFD, 0xF9, 0xB9, 0xB5, 0x8F, 0xBB, 0x3E, 0x8B, 0xE8, 0x21, 0x21, 0x3A, 0xC2, 0xBA, 0x37, 0x24,
  0x61, 0x14, 0x7E, 0xA8, 0x86, 0x0A, 0x5C, 0xD3, 0x5F, 0xA8, 0xDC, 0x3A, 0x06, 0x30, 0x53, 0xFC,
  0x85, 0xBC, 0x60, 0x5C, 0x28, 0x07, 0xCA, 0x12, 0x40, 0xE8, 0x97, 0x6A, 0x63, 0x30, 0xF7, 0x37,
  0x25, 0x76, 0x4D, 0xA9, 0x1A, 0x8C, 0x36, 0xAA, 0xA3, 0x6A, 0x30, 0x68, 0x30, 0x0E, 0x06, 0x03,
  0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x16, 0x06, 0x03,
  0x55, 0x1D, 0x25, 0x04, 0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x4C, 0x09, 0x15, 0x01, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
  0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40, 0x3C, 0x3E, 0x9C,
  0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14,
  0xE8, 0x47, 0x05, 0xA2, 0x59, 0xE9, 0x89, 0x81, 0xF4, 0xC8, 0x0D, 0x4C, 0x0A, 0xD3, 0xD4, 0xDE,
  0x9F, 0xA6, 0xC7, 0x79, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
  0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x8C, 0x39, 0xC6, 0x6C, 0xD6, 0x8F, 0x20, 0xA5,
  0x85, 0x97, 0x27, 0xF9, 0xB5, 0x26, 0xA2, 0xAB, 0xEC, 0x2E, 0x1B, 0x6F, 0x6F, 0x79, 0x9F, 0x8D,
  0xCE, 0x9D, 0x12, 0x23, 0x5F, 0x04, 0xFA, 0xBE, 0x02, 0x21, 0x00, 0xC5, 0xD8, 0x49, 0xD1, 0x40,
  0xE3, 0x2C, 0xDF, 0xC3, 0xD4, 0x11, 0xF9, 0x5E, 0x10, 0x8A, 0x3C, 0x4E, 0xA0, 0xD6, 0x6B, 0x8C,
  0x0E, 0x7B, 0xF1, 0x2B, 0x20, 0x9C, 0x68, 0x53, 0x43, 0x4B, 0x35, 0x31, 0x81, 0xA3, 0x30, 0x81,
  0xA0, 0x02, 0x01, 0x01, 0x30, 0x36, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E,
  0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x02, 0x10, 0x3E, 0xEE, 0x4B, 0xFB,
  0x00, 0x35, 0xC8, 0x96, 0x41, 0xF4, 0xF6, 0x57, 0x16, 0xAD, 0x20, 0xCB, 0x30, 0x0D, 0x06, 0x09,
  0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x07, 0x2A,
  0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x05, 0x00, 0x04, 0x47, 0x30, 0x45, 0x02, 0x21, 0x00, 0xB8,
  0xCC, 0x0F, 0xD2, 0x50, 0x47, 0x3A, 0x11, 0x8D, 0xAE, 0x6C, 0xE9, 0x03, 0x64, 0xA7, 0x44, 0xE6,
  0x0F, 0xF8, 0x6F, 0x7E, 0xE7, 0x61, 0x25, 0x8C, 0x67, 0xE8, 0xC2, 0xD6, 0x33, 0x91, 0x62, 0x02,
  0x20, 0x78, 0xF8, 0x94, 0x0F, 0x46, 0xD2, 0xE2, 0x2B, 0x33, 0xAF, 0xBD, 0x38, 0xA4, 0x43, 0xA6,
  0x6F, 0x8A, 0xBE, 0x26, 0xDC, 0xD0, 0x82, 0x01, 0x3D, 0x00, 0xF3, 0xC2, 0x07, 0x8A, 0xE8, 0xFD,
  0x18,
};

//
// This is TestSignEKUsWith2CertsInSignature.bin.p7b in byte array format.
// It has one EKU in it:
// "1.3.6.1.4.1.311.76.9.21.1"
//
CONST UINT8  TestSignEKUsWith2CertsInSignature[] =
{
  0x30, 0x82, 0x04, 0x61, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x96, 0x30, 0x82, 0x01, 0xAE, 0x30, 0x82, 0x01,
  0x53, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x3E, 0xEE, 0x4B, 0xFB, 0x00, 0x35, 0xC8, 0x96,
  0x41, 0xF4, 0xF6, 0x57, 0x16, 0xAD, 0x20, 0xCB, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x49,
  0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38,
  0x31, 0x35, 0x31, 0x37, 0x33, 0x35, 0x35, 0x30, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38, 0x31,
  0x35, 0x31, 0x37, 0x34, 0x35, 0x35, 0x30, 0x5A, 0x30, 0x23, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03,
  0x55, 0x04, 0x03, 0x0C, 0x18, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73,
  0x69, 0x6E, 0x67, 0x4C, 0x65, 0x61, 0x66, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x30, 0x59, 0x30,
  0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xF6, 0xEA, 0x5E, 0x14, 0x20, 0x6D, 0x11, 0xC0,
  0xFD, 0xF9, 0xB9, 0xB5, 0x8F, 0xBB, 0x3E, 0x8B, 0xE8, 0x21, 0x21, 0x3A, 0xC2, 0xBA, 0x37, 0x24,
  0x61, 0x14, 0x7E, 0xA8, 0x86, 0x0A, 0x5C, 0xD3, 0x5F, 0xA8, 0xDC, 0x3A, 0x06, 0x30, 0x53, 0xFC,
  0x85, 0xBC, 0x60, 0x5C, 0x28, 0x07, 0xCA, 0x12, 0x40, 0xE8, 0x97, 0x6A, 0x63, 0x30, 0xF7, 0x37,
  0x25, 0x76, 0x4D, 0xA9, 0x1A, 0x8C, 0x36, 0xAA, 0xA3, 0x6A, 0x30, 0x68, 0x30, 0x0E, 0x06, 0x03,
  0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x16, 0x06, 0x03,
  0x55, 0x1D, 0x25, 0x04, 0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x4C, 0x09, 0x15, 0x01, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
  0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40, 0x3C, 0x3E, 0x9C,
  0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14,
  0xE8, 0x47, 0x05, 0xA2, 0x59, 0xE9, 0x89, 0x81, 0xF4, 0xC8, 0x0D, 0x4C, 0x0A, 0xD3, 0xD4, 0xDE,
  0x9F, 0xA6, 0xC7, 0x79, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
  0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x8C, 0x39, 0xC6, 0x6C, 0xD6, 0x8F, 0x20, 0xA5,
  0x85, 0x97, 0x27, 0xF9, 0xB5, 0x26, 0xA2, 0xAB, 0xEC, 0x2E, 0x1B, 0x6F, 0x6F, 0x79, 0x9F, 0x8D,
  0xCE, 0x9D, 0x12, 0x23, 0x5F, 0x04, 0xFA, 0xBE, 0x02, 0x21, 0x00, 0xC5, 0xD8, 0x49, 0xD1, 0x40,
  0xE3, 0x2C, 0xDF, 0xC3, 0xD4, 0x11, 0xF9, 0x5E, 0x10, 0x8A, 0x3C, 0x4E, 0xA0, 0xD6, 0x6B, 0x8C,
  0x0E, 0x7B, 0xF1, 0x2B, 0x20, 0x9C, 0x68, 0x53, 0x43, 0x4B, 0x35, 0x30, 0x82, 0x01, 0xE0, 0x30,
  0x82, 0x01, 0x65, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x79, 0x28, 0x1A, 0xE8, 0x7F, 0x21,
  0x4F, 0xBB, 0x47, 0x2F, 0x74, 0x75, 0xE6, 0x98, 0x60, 0xD6, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86,
  0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x0C, 0x16, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E,
  0x67, 0x50, 0x6F, 0x6C, 0x69, 0x63, 0x79, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30,
  0x38, 0x31, 0x35, 0x31, 0x37, 0x33, 0x32, 0x35, 0x34, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38,
  0x31, 0x35, 0x31, 0x37, 0x34, 0x32, 0x35, 0x34, 0x5A, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06,
  0x03, 0x55, 0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72,
  0x73, 0x69, 0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x30, 0x59, 0x30,
  0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x66, 0x41, 0xE6, 0x86, 0x7D, 0x5B, 0x96, 0x6C,
  0x99, 0x2A, 0x82, 0x53, 0x58, 0xA3, 0x6F, 0x59, 0xFC, 0x1C, 0x9F, 0xEB, 0xBE, 0x9D, 0x53, 0xBA,
  0x50, 0x0F, 0x66, 0xBC, 0xE5, 0x90, 0x7E, 0xAA, 0xA2, 0x6C, 0xC2, 0xC9, 0xAA, 0x99, 0x0B, 0x04,
  0xA0, 0x6A, 0xB4, 0xC1, 0xF7, 0x55, 0x60, 0x58, 0xE3, 0xDE, 0x22, 0x57, 0xB1, 0x8A, 0x5C, 0xBC,
  0xDF, 0x22, 0x8A, 0x2B, 0x04, 0x7D, 0x7A, 0x6C, 0xA3, 0x7E, 0x30, 0x7C, 0x30, 0x0E, 0x06, 0x03,
  0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x16, 0x06, 0x03,
  0x55, 0x1D, 0x25, 0x04, 0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x4C, 0x09, 0x15, 0x01, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x08,
  0x30, 0x06, 0x01, 0x01, 0xFF, 0x02, 0x01, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04,
  0x18, 0x30, 0x16, 0x80, 0x14, 0x0B, 0x8A, 0xC7, 0xA8, 0x53, 0x07, 0x40, 0x74, 0x75, 0x0F, 0x85,
  0xCE, 0x5D, 0x5C, 0xC2, 0x23, 0x32, 0xC0, 0xFC, 0xC9, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E,
  0x04, 0x16, 0x04, 0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40,
  0x3C, 0x3E, 0x9C, 0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x03, 0x69, 0x00, 0x30, 0x66, 0x02, 0x31, 0x00, 0xEE, 0x78, 0xDB, 0x74,
  0x56, 0xA3, 0xC2, 0x44, 0x25, 0x0F, 0xA1, 0x89, 0xC0, 0x76, 0xD1, 0xE4, 0xFE, 0xAA, 0xA4, 0xC3,
  0x7D, 0x40, 0xE3, 0x28, 0x1B, 0x0D, 0xAB, 0x32, 0x6B, 0x9A, 0x60, 0x37, 0x2F, 0xB6, 0xF4, 0xCB,
  0xEA, 0xC8, 0x65, 0xB6, 0xDC, 0x24, 0x90, 0x10, 0xFB, 0xD2, 0x42, 0x05, 0x02, 0x31, 0x00, 0x9E,
  0x91, 0xF9, 0x96, 0x8A, 0x41, 0x2C, 0x5E, 0x6B, 0xBC, 0x36, 0x05, 0x3B, 0x50, 0x11, 0x91, 0x9A,
  0xFA, 0x6B, 0x7F, 0x01, 0xF4, 0x6A, 0x7E, 0xCD, 0x98, 0x7E, 0x7D, 0xFD, 0xA8, 0xE3, 0xB2, 0x4F,
  0x71, 0x91, 0xF5, 0xE0, 0x82, 0xA5, 0xE7, 0x5A, 0xE8, 0x19, 0x09, 0xF6, 0x22, 0xBE, 0xC4, 0x31,
  0x81, 0xA3, 0x30, 0x81, 0xA0, 0x02, 0x01, 0x01, 0x30, 0x36, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E,
  0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61,
  0x72, 0x73, 0x69, 0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x02, 0x10,
  0x3E, 0xEE, 0x4B, 0xFB, 0x00, 0x35, 0xC8, 0x96, 0x41, 0xF4, 0xF6, 0x57, 0x16, 0xAD, 0x20, 0xCB,
  0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30,
  0x0B, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x05, 0x00, 0x04, 0x47, 0x30, 0x45,
  0x02, 0x20, 0x4A, 0xFD, 0xAE, 0xDD, 0x5D, 0xC5, 0x86, 0x48, 0xDC, 0xC1, 0xC3, 0xA7, 0xE6, 0x4E,
  0x7F, 0x8B, 0x7B, 0xAB, 0x2C, 0x60, 0xDD, 0x79, 0x4C, 0xF2, 0x45, 0xED, 0x7A, 0xA5, 0x24, 0xE6,
  0x0E, 0x60, 0x02, 0x21, 0x00, 0xAD, 0xCA, 0x3C, 0x29, 0x2F, 0xF9, 0xB1, 0x8E, 0x58, 0x29, 0x3F,
  0x97, 0x20, 0x03, 0x6A, 0x14, 0xDA, 0x83, 0x4A, 0x8F, 0x3E, 0x95, 0x37, 0xC3, 0xF0, 0x12, 0x2A,
  0x7B, 0x24, 0x59, 0x69, 0x27,
};

//
// This is TestSignEKUsWith3CertsInSignature.bin.p7b
// It has one EKU in it:
// "1.3.6.1.4.1.311.76.9.21.1"
//
const UINT8  TestSignEKUsWith3CertsInSignature[] =
{
  0x30, 0x82, 0x06, 0x65, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x05, 0x9B, 0x30, 0x82, 0x01, 0xAE, 0x30, 0x82, 0x01,
  0x53, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x3E, 0xEE, 0x4B, 0xFB, 0x00, 0x35, 0xC8, 0x96,
  0x41, 0xF4, 0xF6, 0x57, 0x16, 0xAD, 0x20, 0xCB, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x49,
  0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38,
  0x31, 0x35, 0x31, 0x37, 0x33, 0x35, 0x35, 0x30, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38, 0x31,
  0x35, 0x31, 0x37, 0x34, 0x35, 0x35, 0x30, 0x5A, 0x30, 0x23, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03,
  0x55, 0x04, 0x03, 0x0C, 0x18, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73,
  0x69, 0x6E, 0x67, 0x4C, 0x65, 0x61, 0x66, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x30, 0x59, 0x30,
  0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xF6, 0xEA, 0x5E, 0x14, 0x20, 0x6D, 0x11, 0xC0,
  0xFD, 0xF9, 0xB9, 0xB5, 0x8F, 0xBB, 0x3E, 0x8B, 0xE8, 0x21, 0x21, 0x3A, 0xC2, 0xBA, 0x37, 0x24,
  0x61, 0x14, 0x7E, 0xA8, 0x86, 0x0A, 0x5C, 0xD3, 0x5F, 0xA8, 0xDC, 0x3A, 0x06, 0x30, 0x53, 0xFC,
  0x85, 0xBC, 0x60, 0x5C, 0x28, 0x07, 0xCA, 0x12, 0x40, 0xE8, 0x97, 0x6A, 0x63, 0x30, 0xF7, 0x37,
  0x25, 0x76, 0x4D, 0xA9, 0x1A, 0x8C, 0x36, 0xAA, 0xA3, 0x6A, 0x30, 0x68, 0x30, 0x0E, 0x06, 0x03,
  0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x16, 0x06, 0x03,
  0x55, 0x1D, 0x25, 0x04, 0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x4C, 0x09, 0x15, 0x01, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
  0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40, 0x3C, 0x3E, 0x9C,
  0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14,
  0xE8, 0x47, 0x05, 0xA2, 0x59, 0xE9, 0x89, 0x81, 0xF4, 0xC8, 0x0D, 0x4C, 0x0A, 0xD3, 0xD4, 0xDE,
  0x9F, 0xA6, 0xC7, 0x79, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
  0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x8C, 0x39, 0xC6, 0x6C, 0xD6, 0x8F, 0x20, 0xA5,
  0x85, 0x97, 0x27, 0xF9, 0xB5, 0x26, 0xA2, 0xAB, 0xEC, 0x2E, 0x1B, 0x6F, 0x6F, 0x79, 0x9F, 0x8D,
  0xCE, 0x9D, 0x12, 0x23, 0x5F, 0x04, 0xFA, 0xBE, 0x02, 0x21, 0x00, 0xC5, 0xD8, 0x49, 0xD1, 0x40,
  0xE3, 0x2C, 0xDF, 0xC3, 0xD4, 0x11, 0xF9, 0x5E, 0x10, 0x8A, 0x3C, 0x4E, 0xA0, 0xD6, 0x6B, 0x8C,
  0x0E, 0x7B, 0xF1, 0x2B, 0x20, 0x9C, 0x68, 0x53, 0x43, 0x4B, 0x35, 0x30, 0x82, 0x01, 0xE0, 0x30,
  0x82, 0x01, 0x65, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x79, 0x28, 0x1A, 0xE8, 0x7F, 0x21,
  0x4F, 0xBB, 0x47, 0x2F, 0x74, 0x75, 0xE6, 0x98, 0x60, 0xD6, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86,
  0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04,
  0x03, 0x0C, 0x16, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E,
  0x67, 0x50, 0x6F, 0x6C, 0x69, 0x63, 0x79, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30,
  0x38, 0x31, 0x35, 0x31, 0x37, 0x33, 0x32, 0x35, 0x34, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38,
  0x31, 0x35, 0x31, 0x37, 0x34, 0x32, 0x35, 0x34, 0x5A, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06,
  0x03, 0x55, 0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72,
  0x73, 0x69, 0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x30, 0x59, 0x30,
  0x13, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x66, 0x41, 0xE6, 0x86, 0x7D, 0x5B, 0x96, 0x6C,
  0x99, 0x2A, 0x82, 0x53, 0x58, 0xA3, 0x6F, 0x59, 0xFC, 0x1C, 0x9F, 0xEB, 0xBE, 0x9D, 0x53, 0xBA,
  0x50, 0x0F, 0x66, 0xBC, 0xE5, 0x90, 0x7E, 0xAA, 0xA2, 0x6C, 0xC2, 0xC9, 0xAA, 0x99, 0x0B, 0x04,
  0xA0, 0x6A, 0xB4, 0xC1, 0xF7, 0x55, 0x60, 0x58, 0xE3, 0xDE, 0x22, 0x57, 0xB1, 0x8A, 0x5C, 0xBC,
  0xDF, 0x22, 0x8A, 0x2B, 0x04, 0x7D, 0x7A, 0x6C, 0xA3, 0x7E, 0x30, 0x7C, 0x30, 0x0E, 0x06, 0x03,
  0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x16, 0x06, 0x03,
  0x55, 0x1D, 0x25, 0x04, 0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37,
  0x4C, 0x09, 0x15, 0x01, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x08,
  0x30, 0x06, 0x01, 0x01, 0xFF, 0x02, 0x01, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04,
  0x18, 0x30, 0x16, 0x80, 0x14, 0x0B, 0x8A, 0xC7, 0xA8, 0x53, 0x07, 0x40, 0x74, 0x75, 0x0F, 0x85,
  0xCE, 0x5D, 0x5C, 0xC2, 0x23, 0x32, 0xC0, 0xFC, 0xC9, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E,
  0x04, 0x16, 0x04, 0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40,
  0x3C, 0x3E, 0x9C, 0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x03, 0x69, 0x00, 0x30, 0x66, 0x02, 0x31, 0x00, 0xEE, 0x78, 0xDB, 0x74,
  0x56, 0xA3, 0xC2, 0x44, 0x25, 0x0F, 0xA1, 0x89, 0xC0, 0x76, 0xD1, 0xE4, 0xFE, 0xAA, 0xA4, 0xC3,
  0x7D, 0x40, 0xE3, 0x28, 0x1B, 0x0D, 0xAB, 0x32, 0x6B, 0x9A, 0x60, 0x37, 0x2F, 0xB6, 0xF4, 0xCB,
  0xEA, 0xC8, 0x65, 0xB6, 0xDC, 0x24, 0x90, 0x10, 0xFB, 0xD2, 0x42, 0x05, 0x02, 0x31, 0x00, 0x9E,
  0x91, 0xF9, 0x96, 0x8A, 0x41, 0x2C, 0x5E, 0x6B, 0xBC, 0x36, 0x05, 0x3B, 0x50, 0x11, 0x91, 0x9A,
  0xFA, 0x6B, 0x7F, 0x01, 0xF4, 0x6A, 0x7E, 0xCD, 0x98, 0x7E, 0x7D, 0xFD, 0xA8, 0xE3, 0xB2, 0x4F,
  0x71, 0x91, 0xF5, 0xE0, 0x82, 0xA5, 0xE7, 0x5A, 0xE8, 0x19, 0x09, 0xF6, 0x22, 0xBE, 0xC4, 0x30,
  0x82, 0x02, 0x01, 0x30, 0x82, 0x01, 0x62, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x41, 0x02,
  0xBA, 0xF9, 0x87, 0x65, 0x6C, 0xB8, 0x4C, 0xD9, 0xA1, 0x04, 0x1C, 0x46, 0x4F, 0xE4, 0x30, 0x0A,
  0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x1D, 0x31, 0x1B, 0x30, 0x19,
  0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x12, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61,
  0x72, 0x73, 0x69, 0x6E, 0x67, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30,
  0x38, 0x31, 0x35, 0x31, 0x37, 0x32, 0x39, 0x32, 0x39, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38,
  0x31, 0x35, 0x31, 0x37, 0x33, 0x39, 0x32, 0x39, 0x5A, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06,
  0x03, 0x55, 0x04, 0x03, 0x0C, 0x16, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72,
  0x73, 0x69, 0x6E, 0x67, 0x50, 0x6F, 0x6C, 0x69, 0x63, 0x79, 0x43, 0x41, 0x30, 0x76, 0x30, 0x10,
  0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22,
  0x03, 0x62, 0x00, 0x04, 0xF9, 0xDC, 0xB9, 0x3F, 0xA8, 0xB5, 0x2B, 0x2E, 0xA7, 0xFC, 0x88, 0xC6,
  0xEF, 0x9C, 0x36, 0x61, 0x3E, 0xFF, 0x8A, 0x5A, 0x52, 0xFF, 0x5A, 0xD0, 0x2C, 0x08, 0xD6, 0x73,
  0x12, 0xF9, 0x62, 0x1E, 0x73, 0x76, 0x83, 0x49, 0xC8, 0xDE, 0xEB, 0x29, 0xA4, 0x78, 0x2A, 0xC1,
  0x30, 0x11, 0x0B, 0x2F, 0x38, 0x00, 0xB7, 0x6B, 0x6B, 0x8E, 0x8C, 0x4B, 0x33, 0x09, 0xF1, 0xEA,
  0xA4, 0x9E, 0x10, 0x4F, 0xC3, 0xBC, 0x30, 0x16, 0x1A, 0x8B, 0xD7, 0x9C, 0x14, 0x11, 0x4E, 0x56,
  0x64, 0x7E, 0x51, 0xE7, 0xDA, 0x26, 0x40, 0xB0, 0xD0, 0xF8, 0xF2, 0xB2, 0x5D, 0x8C, 0x8B, 0x49,
  0x0C, 0x4E, 0x58, 0xB0, 0xA3, 0x63, 0x30, 0x61, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01,
  0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01,
  0x01, 0xFF, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23,
  0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x04, 0xA6, 0xCD, 0x4D, 0xAE, 0x9C, 0x17, 0xDB, 0xC8, 0x13,
  0x57, 0x2D, 0x72, 0x48, 0xAC, 0xBB, 0x6B, 0x1F, 0xFF, 0x61, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D,
  0x0E, 0x04, 0x16, 0x04, 0x14, 0x0B, 0x8A, 0xC7, 0xA8, 0x53, 0x07, 0x40, 0x74, 0x75, 0x0F, 0x85,
  0xCE, 0x5D, 0x5C, 0xC2, 0x23, 0x32, 0xC0, 0xFC, 0xC9, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48,
  0xCE, 0x3D, 0x04, 0x03, 0x02, 0x03, 0x81, 0x8C, 0x00, 0x30, 0x81, 0x88, 0x02, 0x42, 0x01, 0xB4,
  0x75, 0x0B, 0x2F, 0x2A, 0xAA, 0x77, 0x03, 0x08, 0xD9, 0x95, 0x13, 0x2C, 0xE8, 0x47, 0x69, 0x97,
  0xD1, 0x6B, 0xE0, 0x23, 0x94, 0xD5, 0x34, 0x8F, 0x80, 0x20, 0xE3, 0x72, 0x07, 0x7C, 0x6A, 0x52,
  0xE6, 0xF2, 0xC7, 0x85, 0xBD, 0xB6, 0x65, 0x8F, 0x61, 0x73, 0xF9, 0xE7, 0x3A, 0x31, 0x46, 0xC6,
  0xF2, 0xCE, 0xA7, 0x9E, 0xD0, 0xE2, 0xEF, 0x07, 0x59, 0xD0, 0xF1, 0xC4, 0xF6, 0x58, 0x9A, 0x16,
  0x02, 0x42, 0x00, 0x86, 0xE0, 0x25, 0xC1, 0x48, 0xD0, 0x33, 0x30, 0x55, 0xA8, 0x60, 0x44, 0xCA,
  0x47, 0x04, 0x93, 0x46, 0x81, 0xE9, 0x13, 0xE9, 0x67, 0x26, 0x23, 0x2A, 0xBD, 0xEF, 0xC4, 0xCC,
  0x12, 0x99, 0xB3, 0x9E, 0x3D, 0x78, 0xC8, 0x42, 0x06, 0x40, 0x48, 0xCA, 0x3D, 0xCC, 0xD7, 0xA2,
  0x35, 0x37, 0x76, 0x12, 0xDB, 0x9C, 0x90, 0xA4, 0xE4, 0x16, 0xAC, 0x5E, 0xD8, 0x04, 0xF1, 0x59,
  0x66, 0x85, 0xE2, 0x68, 0x31, 0x81, 0xA2, 0x30, 0x81, 0x9F, 0x02, 0x01, 0x01, 0x30, 0x36, 0x30,
  0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74,
  0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E,
  0x67, 0x43, 0x41, 0x02, 0x10, 0x3E, 0xEE, 0x4B, 0xFB, 0x00, 0x35, 0xC8, 0x96, 0x41, 0xF4, 0xF6,
  0x57, 0x16, 0xAD, 0x20, 0xCB, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04,
  0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x05,
  0x00, 0x04, 0x46, 0x30, 0x44, 0x02, 0x20, 0x52, 0x90, 0x02, 0x9E, 0xE8, 0xBF, 0xE8, 0x7E, 0x05,
  0x3B, 0xA3, 0x93, 0xDD, 0x6C, 0x35, 0x31, 0xA3, 0x29, 0x4F, 0x9A, 0x78, 0xEF, 0xD1, 0xAD, 0x07,
  0x37, 0x2D, 0x9A, 0x81, 0xFD, 0x7D, 0xFC, 0x02, 0x20, 0x4C, 0x8A, 0xA4, 0xBD, 0x63, 0x7B, 0x8A,
  0x2C, 0x9E, 0xFC, 0x2A, 0x0F, 0x91, 0xF4, 0x97, 0xB5, 0x3B, 0xC9, 0xD1, 0xCA, 0xEB, 0x02, 0x6A,
  0xB6, 0xFF, 0xE1, 0x15, 0xF5, 0xA2, 0x11, 0xF4, 0xD9,
};

//
// This signature was signed with a leaf that has no EKU's present in the certificate.
//
CONST UINT8  TestSignatureWithNoEKUsPresent[] =
{
  0x30, 0x82, 0x07, 0xDB, 0x02, 0x01, 0x01, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48,
  0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
  0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x06, 0x51, 0x30, 0x82, 0x01, 0xE0, 0x30, 0x82, 0x01,
  0x65, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x79, 0x28, 0x1A, 0xE8, 0x7F, 0x21, 0x4F, 0xBB,
  0x47, 0x2F, 0x74, 0x75, 0xE6, 0x98, 0x60, 0xD6, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE,
  0x3D, 0x04, 0x03, 0x02, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C,
  0x16, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x50,
  0x6F, 0x6C, 0x69, 0x63, 0x79, 0x43, 0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38, 0x31,
  0x35, 0x31, 0x37, 0x33, 0x32, 0x35, 0x34, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38, 0x31, 0x35,
  0x31, 0x37, 0x34, 0x32, 0x35, 0x34, 0x5A, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55,
  0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69,
  0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06,
  0x07, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03,
  0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x66, 0x41, 0xE6, 0x86, 0x7D, 0x5B, 0x96, 0x6C, 0x99, 0x2A,
  0x82, 0x53, 0x58, 0xA3, 0x6F, 0x59, 0xFC, 0x1C, 0x9F, 0xEB, 0xBE, 0x9D, 0x53, 0xBA, 0x50, 0x0F,
  0x66, 0xBC, 0xE5, 0x90, 0x7E, 0xAA, 0xA2, 0x6C, 0xC2, 0xC9, 0xAA, 0x99, 0x0B, 0x04, 0xA0, 0x6A,
  0xB4, 0xC1, 0xF7, 0x55, 0x60, 0x58, 0xE3, 0xDE, 0x22, 0x57, 0xB1, 0x8A, 0x5C, 0xBC, 0xDF, 0x22,
  0x8A, 0x2B, 0x04, 0x7D, 0x7A, 0x6C, 0xA3, 0x7E, 0x30, 0x7C, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D,
  0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x16, 0x06, 0x03, 0x55, 0x1D,
  0x25, 0x04, 0x0F, 0x30, 0x0D, 0x06, 0x0B, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x4C, 0x09,
  0x15, 0x01, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x08, 0x30, 0x06,
  0x01, 0x01, 0xFF, 0x02, 0x01, 0x00, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30,
  0x16, 0x80, 0x14, 0x0B, 0x8A, 0xC7, 0xA8, 0x53, 0x07, 0x40, 0x74, 0x75, 0x0F, 0x85, 0xCE, 0x5D,
  0x5C, 0xC2, 0x23, 0x32, 0xC0, 0xFC, 0xC9, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16,
  0x04, 0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40, 0x3C, 0x3E,
  0x9C, 0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04,
  0x03, 0x02, 0x03, 0x69, 0x00, 0x30, 0x66, 0x02, 0x31, 0x00, 0xEE, 0x78, 0xDB, 0x74, 0x56, 0xA3,
  0xC2, 0x44, 0x25, 0x0F, 0xA1, 0x89, 0xC0, 0x76, 0xD1, 0xE4, 0xFE, 0xAA, 0xA4, 0xC3, 0x7D, 0x40,
  0xE3, 0x28, 0x1B, 0x0D, 0xAB, 0x32, 0x6B, 0x9A, 0x60, 0x37, 0x2F, 0xB6, 0xF4, 0xCB, 0xEA, 0xC8,
  0x65, 0xB6, 0xDC, 0x24, 0x90, 0x10, 0xFB, 0xD2, 0x42, 0x05, 0x02, 0x31, 0x00, 0x9E, 0x91, 0xF9,
  0x96, 0x8A, 0x41, 0x2C, 0x5E, 0x6B, 0xBC, 0x36, 0x05, 0x3B, 0x50, 0x11, 0x91, 0x9A, 0xFA, 0x6B,
  0x7F, 0x01, 0xF4, 0x6A, 0x7E, 0xCD, 0x98, 0x7E, 0x7D, 0xFD, 0xA8, 0xE3, 0xB2, 0x4F, 0x71, 0x91,
  0xF5, 0xE0, 0x82, 0xA5, 0xE7, 0x5A, 0xE8, 0x19, 0x09, 0xF6, 0x22, 0xBE, 0xC4, 0x30, 0x82, 0x02,
  0x01, 0x30, 0x82, 0x01, 0x62, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x41, 0x02, 0xBA, 0xF9,
  0x87, 0x65, 0x6C, 0xB8, 0x4C, 0xD9, 0xA1, 0x04, 0x1C, 0x46, 0x4F, 0xE4, 0x30, 0x0A, 0x06, 0x08,
  0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x1D, 0x31, 0x1B, 0x30, 0x19, 0x06, 0x03,
  0x55, 0x04, 0x03, 0x0C, 0x12, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73,
  0x69, 0x6E, 0x67, 0x52, 0x6F, 0x6F, 0x74, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38, 0x31,
  0x35, 0x31, 0x37, 0x32, 0x39, 0x32, 0x39, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38, 0x31, 0x35,
  0x31, 0x37, 0x33, 0x39, 0x32, 0x39, 0x5A, 0x30, 0x21, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55,
  0x04, 0x03, 0x0C, 0x16, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69,
  0x6E, 0x67, 0x50, 0x6F, 0x6C, 0x69, 0x63, 0x79, 0x43, 0x41, 0x30, 0x76, 0x30, 0x10, 0x06, 0x07,
  0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x02, 0x01, 0x06, 0x05, 0x2B, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62,
  0x00, 0x04, 0xF9, 0xDC, 0xB9, 0x3F, 0xA8, 0xB5, 0x2B, 0x2E, 0xA7, 0xFC, 0x88, 0xC6, 0xEF, 0x9C,
  0x36, 0x61, 0x3E, 0xFF, 0x8A, 0x5A, 0x52, 0xFF, 0x5A, 0xD0, 0x2C, 0x08, 0xD6, 0x73, 0x12, 0xF9,
  0x62, 0x1E, 0x73, 0x76, 0x83, 0x49, 0xC8, 0xDE, 0xEB, 0x29, 0xA4, 0x78, 0x2A, 0xC1, 0x30, 0x11,
  0x0B, 0x2F, 0x38, 0x00, 0xB7, 0x6B, 0x6B, 0x8E, 0x8C, 0x4B, 0x33, 0x09, 0xF1, 0xEA, 0xA4, 0x9E,
  0x10, 0x4F, 0xC3, 0xBC, 0x30, 0x16, 0x1A, 0x8B, 0xD7, 0x9C, 0x14, 0x11, 0x4E, 0x56, 0x64, 0x7E,
  0x51, 0xE7, 0xDA, 0x26, 0x40, 0xB0, 0xD0, 0xF8, 0xF2, 0xB2, 0x5D, 0x8C, 0x8B, 0x49, 0x0C, 0x4E,
  0x58, 0xB0, 0xA3, 0x63, 0x30, 0x61, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF,
  0x04, 0x04, 0x03, 0x02, 0x02, 0x04, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF,
  0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xFF, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18,
  0x30, 0x16, 0x80, 0x14, 0x04, 0xA6, 0xCD, 0x4D, 0xAE, 0x9C, 0x17, 0xDB, 0xC8, 0x13, 0x57, 0x2D,
  0x72, 0x48, 0xAC, 0xBB, 0x6B, 0x1F, 0xFF, 0x61, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04,
  0x16, 0x04, 0x14, 0x0B, 0x8A, 0xC7, 0xA8, 0x53, 0x07, 0x40, 0x74, 0x75, 0x0F, 0x85, 0xCE, 0x5D,
  0x5C, 0xC2, 0x23, 0x32, 0xC0, 0xFC, 0xC9, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D,
  0x04, 0x03, 0x02, 0x03, 0x81, 0x8C, 0x00, 0x30, 0x81, 0x88, 0x02, 0x42, 0x01, 0xB4, 0x75, 0x0B,
  0x2F, 0x2A, 0xAA, 0x77, 0x03, 0x08, 0xD9, 0x95, 0x13, 0x2C, 0xE8, 0x47, 0x69, 0x97, 0xD1, 0x6B,
  0xE0, 0x23, 0x94, 0xD5, 0x34, 0x8F, 0x80, 0x20, 0xE3, 0x72, 0x07, 0x7C, 0x6A, 0x52, 0xE6, 0xF2,
  0xC7, 0x85, 0xBD, 0xB6, 0x65, 0x8F, 0x61, 0x73, 0xF9, 0xE7, 0x3A, 0x31, 0x46, 0xC6, 0xF2, 0xCE,
  0xA7, 0x9E, 0xD0, 0xE2, 0xEF, 0x07, 0x59, 0xD0, 0xF1, 0xC4, 0xF6, 0x58, 0x9A, 0x16, 0x02, 0x42,
  0x00, 0x86, 0xE0, 0x25, 0xC1, 0x48, 0xD0, 0x33, 0x30, 0x55, 0xA8, 0x60, 0x44, 0xCA, 0x47, 0x04,
  0x93, 0x46, 0x81, 0xE9, 0x13, 0xE9, 0x67, 0x26, 0x23, 0x2A, 0xBD, 0xEF, 0xC4, 0xCC, 0x12, 0x99,
  0xB3, 0x9E, 0x3D, 0x78, 0xC8, 0x42, 0x06, 0x40, 0x48, 0xCA, 0x3D, 0xCC, 0xD7, 0xA2, 0x35, 0x37,
  0x76, 0x12, 0xDB, 0x9C, 0x90, 0xA4, 0xE4, 0x16, 0xAC, 0x5E, 0xD8, 0x04, 0xF1, 0x59, 0x66, 0x85,
  0xE2, 0x68, 0x30, 0x82, 0x02, 0x64, 0x30, 0x82, 0x02, 0x0A, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02,
  0x10, 0x2B, 0x7B, 0xDA, 0xCD, 0x87, 0xA0, 0xFD, 0x81, 0x49, 0x5E, 0x0E, 0xDD, 0x4F, 0x81, 0x48,
  0x2D, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31,
  0x20, 0x30, 0x1E, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B,
  0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43,
  0x41, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x38, 0x30, 0x38, 0x32, 0x33, 0x31, 0x39, 0x33, 0x30, 0x30,
  0x32, 0x5A, 0x17, 0x0D, 0x33, 0x38, 0x30, 0x38, 0x32, 0x33, 0x31, 0x39, 0x34, 0x30, 0x30, 0x32,
  0x5A, 0x30, 0x27, 0x31, 0x25, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0C, 0x1C, 0x54, 0x65,
  0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6E, 0x67, 0x4E, 0x6F, 0x45, 0x4B,
  0x55, 0x73, 0x49, 0x6E, 0x53, 0x69, 0x67, 0x6E, 0x65, 0x72, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0D,
  0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
  0x0F, 0x00, 0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xA5, 0x9B, 0x95, 0xF0, 0x63,
  0x88, 0xC3, 0x74, 0x26, 0x08, 0x68, 0x25, 0xAA, 0x16, 0xBB, 0x2A, 0x15, 0xC1, 0xA2, 0xAF, 0x72,
  0x06, 0x51, 0x37, 0xE1, 0x4F, 0x6D, 0x6A, 0xFC, 0x17, 0xCE, 0x82, 0x7B, 0x79, 0xFF, 0x2A, 0x84,
  0x0F, 0xCB, 0xA9, 0x65, 0x70, 0xC9, 0xF9, 0xE7, 0x61, 0x90, 0x28, 0x03, 0x6E, 0x41, 0x25, 0xFF,
  0xC2, 0x3A, 0x60, 0x1A, 0x42, 0x37, 0x80, 0x9F, 0x6B, 0x99, 0x11, 0x5A, 0x1F, 0xAE, 0x7B, 0x3F,
  0xF3, 0x2B, 0x45, 0xDD, 0xDA, 0xC4, 0x32, 0xD7, 0xD8, 0xB5, 0x3D, 0x46, 0x0C, 0x8A, 0x11, 0xEE,
  0x94, 0x8C, 0xB5, 0x69, 0xD3, 0x91, 0x4F, 0x4E, 0xE5, 0xBB, 0x93, 0xDD, 0x1E, 0xA7, 0x20, 0x03,
  0xFD, 0xD6, 0x8F, 0x76, 0x91, 0x9A, 0x2B, 0x41, 0x91, 0xD8, 0x35, 0x7A, 0x0D, 0x80, 0xB9, 0xD4,
  0xD2, 0x56, 0xF1, 0xF9, 0xC5, 0xE6, 0xF2, 0xAE, 0x10, 0x7E, 0xD7, 0x6C, 0x40, 0x86, 0x37, 0x99,
  0x85, 0xF4, 0xCA, 0x1D, 0x6D, 0x90, 0x63, 0xF7, 0x63, 0xC6, 0x00, 0xA1, 0xA9, 0x79, 0x7D, 0x88,
  0x05, 0xA2, 0x92, 0x64, 0xA7, 0x8E, 0xA8, 0x9E, 0xC3, 0x1F, 0x63, 0xEE, 0x6C, 0xE3, 0x9B, 0xA6,
  0xD7, 0x9D, 0x9C, 0x9F, 0x0D, 0x7A, 0xF7, 0x00, 0x92, 0x16, 0xE8, 0x64, 0x36, 0x07, 0x66, 0x8E,
  0x6E, 0x51, 0x09, 0x3F, 0x2A, 0xD6, 0x6D, 0x4E, 0xCC, 0x49, 0xE1, 0xC3, 0x59, 0x63, 0xDC, 0x51,
  0xAA, 0xD1, 0xE7, 0x42, 0x2F, 0x72, 0x25, 0x46, 0x83, 0x7D, 0x23, 0xCE, 0xF7, 0x96, 0x2F, 0x04,
  0xE8, 0xAE, 0xF6, 0x19, 0x6D, 0x2D, 0xB2, 0x64, 0xDB, 0x69, 0x33, 0x96, 0x58, 0xA0, 0x6C, 0x77,
  0xF0, 0xD0, 0x5E, 0x36, 0xB1, 0x28, 0x89, 0x83, 0xB6, 0xA5, 0x13, 0x0E, 0x6B, 0x6E, 0x73, 0x2C,
  0x6F, 0xDA, 0xF9, 0x6C, 0x11, 0x4F, 0xCC, 0xF7, 0x89, 0x77, 0x83, 0x02, 0x03, 0x01, 0x00, 0x01,
  0xA3, 0x52, 0x30, 0x50, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04, 0x04,
  0x03, 0x02, 0x01, 0x86, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
  0x14, 0xA4, 0x21, 0xA5, 0x99, 0x40, 0x41, 0xF7, 0x1B, 0xFF, 0x35, 0xD0, 0x40, 0x3C, 0x3E, 0x9C,
  0x7C, 0xD1, 0xCC, 0xD1, 0xD1, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14,
  0xB6, 0xAF, 0x0D, 0xC1, 0xBE, 0xD1, 0xE4, 0x21, 0x37, 0xFF, 0xAC, 0x7D, 0x2D, 0xD4, 0x51, 0x8A,
  0x61, 0x95, 0x50, 0x7F, 0x30, 0x0A, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x04, 0x03, 0x02,
  0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x74, 0x7D, 0xF5, 0xAF, 0xC3, 0xA7, 0x06, 0x99, 0x1F,
  0x2C, 0xAF, 0x69, 0xAA, 0x40, 0x97, 0xE9, 0xF8, 0x4C, 0xD2, 0x71, 0x17, 0x25, 0x85, 0xB7, 0xA6,
  0x97, 0x61, 0x89, 0x82, 0xED, 0xF0, 0xF6, 0x02, 0x21, 0x00, 0xA6, 0xF4, 0x59, 0x85, 0x53, 0x31,
  0xEF, 0x6F, 0x37, 0xB6, 0x8E, 0xA2, 0xDE, 0xC3, 0xFD, 0xD6, 0xFB, 0x50, 0x63, 0x4E, 0xB4, 0xBB,
  0xC1, 0xB1, 0x07, 0x82, 0x80, 0x14, 0x28, 0x6A, 0x94, 0x94, 0x31, 0x82, 0x01, 0x61, 0x30, 0x82,
  0x01, 0x5D, 0x02, 0x01, 0x01, 0x30, 0x36, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55,
  0x04, 0x03, 0x0C, 0x17, 0x54, 0x65, 0x73, 0x74, 0x45, 0x4B, 0x55, 0x50, 0x61, 0x72, 0x73, 0x69,
  0x6E, 0x67, 0x49, 0x73, 0x73, 0x75, 0x69, 0x6E, 0x67, 0x43, 0x41, 0x02, 0x10, 0x2B, 0x7B, 0xDA,
  0xCD, 0x87, 0xA0, 0xFD, 0x81, 0x49, 0x5E, 0x0E, 0xDD, 0x4F, 0x81, 0x48, 0x2D, 0x30, 0x0D, 0x06,
  0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x0D, 0x06, 0x09,
  0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82, 0x01, 0x00, 0x39,
  0xEC, 0xA8, 0x00, 0x4F, 0x7E, 0x2C, 0x3C, 0x48, 0xDF, 0x6F, 0x6A, 0x37, 0x69, 0xA4, 0xB1, 0xEB,
  0x7D, 0xD2, 0x3B, 0xC5, 0xEE, 0x57, 0x1F, 0x8F, 0xCA, 0x60, 0x80, 0x88, 0x24, 0x29, 0xB5, 0x00,
  0x28, 0xFF, 0x8F, 0x0A, 0x1E, 0xB0, 0x9E, 0x90, 0x09, 0xF1, 0x46, 0xC1, 0x53, 0xE8, 0x27, 0x29,
  0xFA, 0xFE, 0xA3, 0x84, 0x79, 0xB2, 0x27, 0x46, 0x9B, 0xD8, 0x33, 0xBD, 0x8F, 0x41, 0x06, 0xEE,
  0xD6, 0xE7, 0x49, 0xAF, 0x48, 0xD0, 0xD4, 0xF1, 0x06, 0xCC, 0xD3, 0x26, 0x47, 0xAE, 0x03, 0x7A,
  0xA5, 0xAF, 0xDE, 0x7F, 0x57, 0x32, 0x51, 0x49, 0x8C, 0x18, 0xD2, 0x4D, 0x50, 0x27, 0xAC, 0x7C,
  0x33, 0xB8, 0xB8, 0x1C, 0xF8, 0x0C, 0x49, 0xEB, 0xF5, 0xD7, 0xA0, 0x35, 0x77, 0x61, 0xC0, 0x60,
  0xAF, 0x23, 0x8C, 0xCC, 0xB0, 0x58, 0xB3, 0x08, 0x89, 0x2B, 0xC0, 0x61, 0x0C, 0xA6, 0x40, 0x7A,
  0x37, 0x6D, 0x6D, 0x44, 0xFF, 0xF1, 0xCE, 0xDE, 0x7A, 0x0C, 0xC4, 0x51, 0x61, 0xE6, 0x72, 0x6B,
  0xA9, 0x6A, 0x7B, 0xCF, 0xAA, 0x9A, 0x28, 0x74, 0x87, 0x30, 0xBA, 0xED, 0x40, 0xF0, 0x75, 0x2B,
  0xD7, 0xE6, 0x9F, 0xDD, 0x6B, 0xB7, 0xEA, 0x08, 0x0D, 0x9F, 0xA3, 0x25, 0x27, 0xB7, 0xBF, 0x41,
  0xC2, 0x64, 0xB5, 0x6F, 0x8F, 0xB1, 0x5F, 0xE5, 0xDF, 0xCA, 0x16, 0x04, 0x61, 0xFF, 0x74, 0x25,
  0xEE, 0xF6, 0x69, 0xF1, 0x7F, 0xE5, 0x97, 0x72, 0x58, 0xD8, 0x9F, 0xA5, 0x19, 0x32, 0x46, 0x95,
  0x67, 0x63, 0x23, 0x54, 0x46, 0x05, 0x15, 0xC3, 0x3E, 0x04, 0x7E, 0x88, 0xD0, 0xB5, 0x52, 0xF0,
  0x38, 0x3C, 0x8B, 0xD4, 0x3D, 0x9A, 0x55, 0x1B, 0x14, 0xE5, 0x37, 0x8D, 0xB7, 0x6C, 0x39, 0x91,
  0x23, 0x31, 0x63, 0x89, 0x7C, 0x75, 0x02, 0x79, 0xCC, 0x95, 0x58, 0x99, 0x3A, 0xDB, 0xF5,
};
