/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * do_key.c: Process key strokes.
 *
 * This file defines and implements the keyboard user interface.
 *
 * For brevity, this source file is written upside-down:
 * first, lots of little functions to perform all the possible operations,
 * then a table of key-values and which functions they call for the four
 * possible combinations of Shift and Ctrl,
 * then the public function, do_key(), which makes use of all the above.
 */

#include "spettro.h"
#include "do_key.h"

/*
 * Local header files
 */
#include "audio.h"
#include "a_cache.h"
#include "axes.h"
#include "barlines.h"
#include "colormap.h"
#include "convert.h"
#include "dump.h"
#include "gui.h"
#include "key.h"
#include "overlay.h"
#include "paint.h"
#include "r_cache.h"
#include "schedule.h"
#include "ui.h"
#include "ui_funcs.h"
#include "window.h"

#include <libgen.h>	/* for basename() */

static void
k_cycle_colors(key_t key)
{
    change_colormap();
    repaint_display(TRUE);
}

static void
k_quit(key_t key)
{
    quitting = TRUE;
    gui_quit_main_loop();
}

static void
k_next(key_t key)
{
    nexting = TRUE;
    gui_quit_main_loop();
}

static void
k_space(key_t key)	/* Play/Pause/Next_file/Quit */
{
    switch (playing) {
    case PLAYING:
	pause_audio();
	break;

    case STOPPED:
stopped:
	/* Replay from the beginning */
	set_playing_time(0.0);
	start_playing();

	break;

    case PAUSED:
	/* The SDL audio system sometimes stops at 9.99998 instead of 10,
	 * so if it's paused within a thounsandth of a second from the end
	 * that's why. */
	if (get_playing_time() + 0.001 >= audio_file_length()) {
	    /* They went "End" while it was paused. Do the same as STOPPED. */
	    goto stopped;
	}
	continue_playing();
	break;
    }
}

/* Play/Pause key - the same as Space */
static void
k_play(key_t key)
{
    k_space(key);
}

static void
k_stop(key_t key)
{
    if (playing == PLAYING) pause_audio();
}

/*
 * Arrow <-/->: Jump back/forward a tenth of a screenful
 * With Shift, a whole screenful. With Ctrl one pixel.
 * With Ctrl-Shift, one second.
 */
static void
k_left_right(key_t key)
{
    double by;

    if (!Shift && !Ctrl) by = disp_width * secpp / 10;
    if (Shift && !Ctrl) by = disp_width * secpp;
    if (!Shift && Ctrl) by = secpp;
    if (Shift && Ctrl) by = 1.0;
    time_pan_by(key == KEY_LEFT ? -by : +by);
}

/*
 * Home and End: Go to start or end of piece
 */
static void
k_home(key_t key)
{
    set_playing_time(0.0);
    if (playing == STOPPED) playing = PAUSED;
}

static void
k_end(key_t key)
{
    /* Go to the last sample */
    set_playing_time(audio_file_length());
    stop_playing();
}

/*
 * Arrow Up/Down: Pan the frequency axis by a tenth of the screen height.
 * With Shift: by a screenful. With Ctrl, by a pixel. With both, by a semitone.
 * The argument to freq_pan_by() multiplies min_freq and max_freq.
 * Page Up/Down: Pan the frequency axis by a screenful
 */
static void
k_freq_pan(key_t key)
{
    switch (key) {
    case KEY_UP:
	freq_pan_by((Ctrl && !Shift) ? v_pixel_freq_ratio():
		    (Shift && !Ctrl) ? max_freq / min_freq :
		    (Shift && Ctrl) ? pow(2, 1/12.0) :
		    pow(max_freq / min_freq, 1/10.0));
	break;
    case KEY_DOWN:
	freq_pan_by((Ctrl && !Shift) ? 1.0 / v_pixel_freq_ratio() :
		    (Shift && !Ctrl) ? min_freq / max_freq :
		    (Shift && Ctrl) ? 1.0 / pow(2, 1/12.0) :
		    pow(min_freq / max_freq, 1/10.0));
	break;
    case KEY_PGUP:
	freq_pan_by(max_freq/min_freq);
	break;
    case KEY_PGDN:
	freq_pan_by(min_freq/max_freq);
	break;
    default:
	break;	/* Shut up compiler warnings */
    }

    gui_update_display();
}

/* Zoom on the time axis by a factor of two so that, when zooming in,
 * half of the results are still valid
 */
static void
k_time_zoom(key_t key)
{
    time_zoom_by(Shift ? 2.0 : 0.5);
    repaint_display(FALSE);
}

/* Y/y: Zoom in/out on the frequency axis.
 * With Shift, zoom in, without Shift, zoom out
 * Without Ctrl, by a factor of two; with Ctrl by one pixel top and bottom.
 */
static void
k_freq_zoom(key_t key)
{
    double by;	/* How much to zoom in by: >1 = zoom in, <1 = zoom out */

    if (Ctrl) by = (double)(max_y - min_y) / (double)((max_y-1) - (min_y+1));
    else by = 2.0;
    freq_zoom_by(Shift ? by : 1.0/by);
    repaint_display(TRUE);
}

/* Capital letters choose the window function */
static void
k_set_window(key_t key)
{
    window_function_t new_fn;

    switch (key) {
    case KEY_K: new_fn = KAISER;	break;
    case KEY_N: new_fn = NUTTALL;	break;
    case KEY_H: new_fn = HANN;		break;
    case KEY_B: new_fn = BLACKMAN;	break;
    case KEY_D: new_fn = DOLPH;		break;
    default:
	fprintf(stderr, "Internal error: Impossible window key %d\n", key);
	return;
    }

    if (new_fn != window_function) {
	window_function = new_fn;
	if (show_time_axes) draw_status_line();
	drop_all_work();
	repaint_display(TRUE);
    }
}

/* Z and z change the dynamic range of color spectrum by 6dB.
 * 'z' brightens the dark areas, which is achieved by reducing
 * the dynamic range;
 * 'Z' darkens them to reduce visibility of background noise.
 * With Ctrl held, they brighten/darken by 1dB.
 */
static void
k_contrast(key_t key)
{
    float by = Ctrl ? 1.0 : 6.0;

    change_dyn_range(Shift ? -by : +by);
    repaint_display(TRUE);
}
/* I and i increase/decrease the brightness of the graph by changing the
 * loudness that is represented by the brightest pixel color.
 */
static void
k_brightness(key_t key)
{
    float by = Ctrl ? 1.0/6.0 : 1.0;

    /* Shift means "more brightness", which means diminishing logmax */
    change_logmax(Shift ? -by : by);
    repaint_display(TRUE);
}

/* Toggle frequency or time axis. */
static void
k_toggle_axes(key_t key)
{
    if (!Shift) {	/* 'a': toggle frequency axes */
	/* disp_offset can change if it's set by DISP_OFFSET and the value is
	 * within the frequency axes */
	int old_disp_offset = disp_offset;
	show_frequency_axes = !show_frequency_axes;
	set_disp_offset();
	/* If removing frequency axes, repaint where they were */
	if (!show_frequency_axes) {
	    repaint_columns(0, frequency_axis_width - 1, min_y, max_y, FALSE);
	    repaint_columns(max_x - note_name_axis_width, max_x, min_y, max_y, FALSE);
	}
	if (disp_offset != old_disp_offset) repaint_display(FALSE);
    } else {		/* 'A': Toggle time axis and status line */
	show_time_axes = !show_time_axes;

	/* Adding/removing the top and bottom axes scales the graph vertically
	 * so repaint. We only need to repaint unpainted columns if we are
	 * removing the axes, so that background color columns and vertical
	 * overlays overwrite where the axes were. */
	repaint_columns(min_x, max_x, min_y, max_y, show_time_axes);
    }
    draw_axes();
}

/* w: Cycle through window functions;
 * W: cycle backwards
 */
static void
k_cycle_window(key_t key)
{
    window_function = ((!Shift) ? (window_function + 1)
			        : (window_function + NUMBER_OF_WINDOW_FUNCTIONS-1))
		      % NUMBER_OF_WINDOW_FUNCTIONS;
    if (show_time_axes) draw_status_line();
    drop_all_work();
    repaint_display(FALSE);
}

/* Toggle staff/piano/guitar line overlays
 *
 * At present we force a repaint of all displayed columns with the new
 * overlay. It would be more efficient, when adding overlaid lines,
 * just to draw the horizontal lines on the display for the columns
 * already displaying spectral data, and when removing them just ask for
 * those few pixels to be refreshed from the spectral data.
 */
static void
k_overlay(key_t key)
{
    switch (key) {
    case KEY_K:
	/* Cycle white keys through white-green-none */
	piano_lines = !piano_lines;
	break;
    case KEY_S:
	staff_lines = !staff_lines;
	if (staff_lines) guitar_lines = FALSE;
	break;
    case KEY_G:
	guitar_lines = !guitar_lines;
	if (guitar_lines) staff_lines = FALSE;
	break;
    default:	/* Silence compiler warnings */
	break;
    }
    make_row_overlay();
    repaint_display(TRUE);
}

/* Make a screen dump */
static void
k_screendump(key_t key)
{
    dump_screenshot();
}

/* Make an audio dump of the sound between the bar lines.
 * Includes the sample at left_bar_time but not the one at right_bar_time */
static void
k_audiodump(key_t key)
{
    secs_t start_time, end_time;
    int start_frame, end_frame;
    int nframes;
    freq_t sample_rate = sr;
    char *filestem = "spettro";		/* Start of output filename */
    char *s;				/* Whole output filename */
    /* basename() may modify the string, so work on a copy */
    char *fn = strdup(current_audio_file()->filename);
    char *bn = basename(fn);

    start_time = left_bar_time == UNDEFINED
		 ? 0.0 : left_bar_time;
    end_time   = right_bar_time == UNDEFINED
		 ? audio_file_length() : right_bar_time;

    /* Allow for left bar line to the right of right bar line */
    if (start_time > end_time) {
	secs_t tmp = start_time;
	start_time = end_time;
	end_time = tmp;
    }
    start_frame = round(start_time * sample_rate);
    end_frame = round(end_time * sample_rate);
    nframes = end_frame - start_frame;

    /* Allocate space for output filename, with a few bytes slop for the
     * removed suffix of the original file name.
     * Maximum length of secs is 11 for 99:59:59.99
     */
    /*                 spettro   -l  secs -r  secs  filename  .wav \0 */
    s = Malloc(strlen(filestem)+1+2+1+11+1+2+1+11+1+strlen(bn)+ 4 + 1);

    /* Construct the output file name */
    strcpy(s, filestem);
#define add(s, fmt, val) sprintf(s + strlen(s), fmt, val)
    if (left_bar_time != UNDEFINED) add(s, " -l %s",
    	seconds_to_string(left_bar_time));
    if (right_bar_time != UNDEFINED) add(s, " -r %s",
    	seconds_to_string(right_bar_time));
    {
	char *dot = strrchr(bn, '.');

	if (dot) *dot = '\0';
	add(s, " %s.wav", bn);
    }
#undef add

    dump_to_audio_file(s, start_frame, nframes);
    free(s);
    free(fn);
}

/* Print current UI parameters and derived values */
static void
k_print_params(key_t key)
{
    audio_file_t *af = current_audio_file();
    static bool first_time = FALSE;

    if (!first_time) {
	int i;

	for (i=0; i<79; i++) putchar('-');
	putchar('\n');
    } else
	first_time = FALSE;

    printf("filename=\"%s\" opened with %s\n", af->filename,
#if HAVE_LIBMPG123
	   af->opened_with == libmpg123 ? "libmpg123" :
#endif
#if HAVE_LIBSNDFILE
	   af->opened_with == libsndfile ? "libsndfile" :
#endif
#if HAVE_LIBAV
	   af->opened_with == libav ? "libav" :
#endif
	   "nothing");
    printf("channels=%d sample_rate=%g audio_length=%s (%ld frames)\n",
	   af->channels, af->sample_rate,
	   seconds_to_string(audio_file_length()), (long)af->frames);

    printf("disp_width=%d disp_height=%d min_x=%d max_x=%d min_y=%d max_y=%d\n",
	    disp_width,   disp_height,   min_x,   max_x,   min_y,   max_y);

    printf(
"min_freq=%g max_freq=%g dyn_range=%gdB logmax=%.3g fft_freq=%g speclen=%ld\n",
 min_freq,   max_freq,   dyn_range,     logmax,     fft_freq,
	fft_freq_to_speclen(fft_freq, sr));

    printf("window=%s disp_time=%.2f disp_offset=%d ppsec=%.3f fps=%.3f\n",
	window_name(window_function), disp_time, disp_offset, ppsec, fps);
    printf("softvol=%.2f softvol_double=%s\n",
	softvol, seconds_to_string(softvol_double));

    printf("%s %s ",
	playing == PLAYING ? "Playing" :
	playing == STOPPED ? "Stopped at" :
	playing == PAUSED  ? "Paused at" : "Doing what? at",
	seconds_to_string(get_playing_time()));
    printf("(player: %s) ",
	seconds_to_string(get_audio_players_time()));
    printf("Showing %s ",
	seconds_to_string(screen_column_to_start_time(min_x)));
    printf("to %s\n",
	seconds_to_string(screen_column_to_start_time(max_x + 1)));
    if (left_bar_time != UNDEFINED)
	printf("left bar=%.3f", left_bar_time);
    if (right_bar_time != UNDEFINED) {
	if (left_bar_time != UNDEFINED) printf(" ");
	printf("right bar=%.3f", right_bar_time);
    }
    if (left_bar_time != UNDEFINED && right_bar_time != UNDEFINED)
	printf(" interval=%.3f beats_per_bar=%d bpm=%ld",
	       fabs(right_bar_time - left_bar_time), beats_per_bar,
	       (long) round(60.0 / fabs(right_bar_time - left_bar_time)
			       * beats_per_bar));
    if (left_bar_time != UNDEFINED || right_bar_time != UNDEFINED)
	printf("\n");
}

static void
k_print_cache_stats(key_t key)
{
    print_audio_cache_stats();
    print_result_cache_stats();
}

/* Flip fullscreen mode */
static void
k_fullscreen(key_t key)
{
    gui_fullscreen();
}

static void
k_set_autoplay(key_t key)
{
    autoplay_mode = TRUE;
}

static void
k_clear_autoplay(key_t key)
{
    autoplay_mode = FALSE;
}

/* Change FFT size: with Shift, double the sample size, without halve it */
static void
k_fft_size(key_t key)
{
    if (Shift) {
	/* Increase FFT size; decrease FFT frequency */
	if (DELTA_EQ(fft_freq, MIN_FFT_FREQ)) {
	    /* Already at the minimum value: do nothing */
	    return;       
	}
	fft_freq /= 2;
	if (DELTA_LT(fft_freq, MIN_FFT_FREQ)) fft_freq = MIN_FFT_FREQ;
    } else {
	/* Decrease FFT size: increase FFT frequency */
	fft_freq *= 2;
	/* Apply maximum values */
	if (fft_freq > sr / 4) fft_freq = sr / 4;
    }
    reposition_audio_cache();
    drop_all_work();

    if (show_time_axes) draw_status_line();

    /* Any calcs that are currently being performed will deliver
     * a result for the old speclen, which will be ignored (or cached)
     */
    repaint_display(FALSE);
}

/* Change the position of the green line */
static void
k_disp_offset(key_t key)
{
    green_line_off = TRUE;
    repaint_column(disp_offset, 0, max_y, FALSE);
    if (Shift) {
	if (disp_offset < max_x) disp_offset++;
    } else {
	if (disp_offset > min_x) disp_offset--;
    }
    green_line_off = FALSE;
    repaint_column(disp_offset, 0, max_y, FALSE);
    gui_update_display();
}

/* Set left or right bar line position to current play position */
static void
k_left_barline(key_t key)
{
    set_left_bar_time(disp_time + secpp / 2);
}

static void
k_right_barline(key_t key)
{
    set_right_bar_time(disp_time + secpp / 2);
}

static void
k_no_barlines(key_t key)
{
    set_left_bar_time(UNDEFINED);
    set_right_bar_time(UNDEFINED);
}

/* Refresh the display from the result cache and redrawing bar/beat lines */
static void
k_refresh(key_t key)
{
    repaint_display(FALSE);
}

/* Refresh the display recalculating all results from the audio */
static void
k_redraw(key_t key)
{
    drop_all_work();
    drop_all_results();
    repaint_display(FALSE);
}

/* softvol volume controls */
static void
k_softvol_down(key_t key)
{
    audio_set_softvol(softvol * 0.9);
}

static void
k_softvol_up(key_t key)
{
    /* Without this, if they start with -v 0 they can never raise it */
    if (DELTA_LE(softvol, 0.0)) audio_set_softvol(DELTA);

    audio_set_softvol(softvol / 0.9);
}

/* Beats per bar */
static void
k_beats_per_bar(key_t key)
{
    /* Under SDL2, Shift-number gives the punctuation characters above them
     * and Shift-Ctrl-number gives both Ctrl+Shift+N and the punctuation char.
     * Under SDL2, Ctrl-F1 to F12 seem not to be delivered. Shift-Fn yes.
     */
    switch (key) {
    case KEY_1: case KEY_F1:	set_beats_per_bar(1);	break;
    case KEY_2: case KEY_F2:	set_beats_per_bar(2);	break;
    case KEY_3: case KEY_F3:	set_beats_per_bar(3);	break;
    case KEY_4: case KEY_F4:	set_beats_per_bar(4);	break;
    case KEY_5: case KEY_F5:	set_beats_per_bar(5);	break;
    case KEY_6: case KEY_F6:	set_beats_per_bar(6);	break;
    case KEY_7: case KEY_F7:	set_beats_per_bar(7);	break;
    case KEY_8: case KEY_F8:	set_beats_per_bar(8);	break;
    case KEY_9:	case KEY_F9:	set_beats_per_bar(9);	break;
		case KEY_F10:	set_beats_per_bar(10);	break;
		case KEY_F11:	set_beats_per_bar(11);	break;
		case KEY_F12:	set_beats_per_bar(12);	break;
    default:
	fprintf(stderr, "Internal error: Impossible beats-per-bar key %d\n", key);
	return;
    }
}

static void
k_dump_audio_cache(key_t key)
{
    dump_audio_cache();
}

/* Now for the table of key-to-function mappings */

typedef struct {
    key_t key;
    char *name;
    void (*plain)(key_t key);
    void (*shift)(key_t key);
    void (*ctrl)(key_t key);
    void (*shiftctrl)(key_t key);
} key_fn;

static void
k_none(key_t key)
{
}

/* k_bad() needs key_fns[]; key_fns[] needs k_bad() */
static key_fn key_fns[];

static void
k_bad(key_t key)
{
    fprintf(stderr, "%s%s%s doesn't do anything\n",
	Ctrl  ? "Ctrl-"  : "",
	Shift ? "Shift-" : "",
	key_fns[key].name);
}

/* This must have the same entries in the same order as "enum key" in key.h */
static key_fn key_fns[] = {
    /* KEY	Name    plain		Shift		Ctrl		ShiftCtrl */
    { KEY_NONE,	"None", k_none,		k_none,		k_none,		k_none },
    { KEY_Q,	"Q",	k_quit,		k_quit,		k_quit,		k_quit },
    { KEY_C,	"C",    k_cycle_colors,	k_bad,		k_bad,		k_bad },
    { KEY_I,	"I",    k_brightness,	k_brightness,	k_brightness,	k_brightness },
    { KEY_Z,	"Z",    k_contrast,	k_contrast,	k_contrast,	k_contrast },
    { KEY_ESC,	"Esc",  k_bad,		k_bad,		k_bad,		k_bad },
    { KEY_SPACE,"Space",k_space,	k_bad,		k_bad,		k_bad },
    { KEY_LEFT, "Left", k_left_right,	k_left_right,	k_left_right,	k_left_right },
    { KEY_RIGHT,"Right",k_left_right,	k_left_right,	k_left_right,	k_left_right },
    { KEY_HOME,	"Home",	k_home,		k_bad,		k_bad,		k_bad },
    { KEY_END,	"End",  k_end,		k_bad,		k_bad,		k_bad },
    { KEY_UP,	"Up",   k_freq_pan,	k_freq_pan,	k_freq_pan,	k_freq_pan },
    { KEY_DOWN,	"Down", k_freq_pan,	k_freq_pan,	k_freq_pan,	k_freq_pan },
    { KEY_PGUP,	"PageUp",  k_freq_pan,	k_bad,		k_bad,		k_bad },
    { KEY_PGDN,	"PageDown",k_freq_pan,	k_bad,		k_bad,		k_bad },
    { KEY_X,	"X",    k_time_zoom,	k_time_zoom,	k_bad,		k_bad },
    { KEY_Y,	"Y",    k_freq_zoom,	k_freq_zoom,	k_freq_zoom,	k_freq_zoom },
    { KEY_PLUS,	"Plus", k_softvol_up,	k_bad,		k_bad,		k_bad },
    { KEY_MINUS,"Minus",k_softvol_down,	k_bad,		k_bad,		k_bad },
    { KEY_EQUALS,"Equals",k_bad,	k_bad,		k_bad,		k_bad },
    { KEY_STAR, "Star", k_bad,		k_bad,		k_bad,		k_bad },
    { KEY_SLASH,"slash",k_bad,		k_bad,		k_bad,		k_bad },
    { KEY_K,	"K",    k_overlay,	k_bad,		k_set_window,	k_bad },
    { KEY_S,	"S",    k_overlay,	k_bad,		k_bad,		k_bad },
    { KEY_G,	"G",    k_overlay,	k_bad,		k_bad,		k_bad },
    { KEY_O,	"O",    k_screendump,	k_bad,		k_audiodump,	k_dump_audio_cache },
    { KEY_P,	"P",    k_clear_autoplay,k_set_autoplay,k_print_params,	k_print_cache_stats },
    { KEY_T,	"T",    k_bad,		k_bad,		k_bad,		k_bad },
    { KEY_F,	"F",    k_fft_size,	k_fft_size,	k_fullscreen,	k_bad },
    { KEY_L,	"L",    k_left_barline,	k_bad,		k_refresh,	k_bad },
    { KEY_R,	"R",    k_right_barline,k_bad,		k_redraw,	k_bad },
    { KEY_B,	"B",    k_bad,		k_bad,		k_set_window,	k_bad },
    { KEY_D,	"D",    k_disp_offset,	k_disp_offset,	k_set_window,	k_bad },
    { KEY_A,	"A",    k_toggle_axes,	k_toggle_axes,	k_bad,		k_bad },
    { KEY_W,	"W",    k_bad,		k_bad,		k_cycle_window,	k_cycle_window },
    { KEY_M,	"M",    k_bad,		k_bad,		k_bad,		k_bad },
    { KEY_H,	"H",	k_bad,		k_bad,		k_set_window,	k_bad },
    { KEY_N,	"N",    k_next,		k_bad,		k_set_window,	k_bad },
    { KEY_0,	"0",	k_no_barlines,	k_bad,		k_bad,		k_bad },
    { KEY_1,	"1",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_2,	"2",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_3,	"3",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_4,	"4",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_5,	"5",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_6,	"6",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_7,	"7",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_8,	"8",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_9,	"9",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F1,	"F1",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F2,	"F2",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F3,	"F3",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F4,	"F4",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F5,	"F5",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F6,	"F6",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F7,	"F7",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F8,	"F8",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F9,	"F9",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F10,	"F10",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F11,	"F11",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    { KEY_F12,	"F12",	k_beats_per_bar,k_bad,		k_bad,		k_bad },
    /* Extended keyboard's >/|| [] |<< and >>| buttons */
    { KEY_PLAY,	"Play",	k_play,		k_bad,		k_bad,		k_bad },
    { KEY_STOP,	"Stop",	k_stop,		k_bad,		k_bad,		k_bad },
    { KEY_PREV,	"Prev",	k_bad,		k_bad,		k_bad,		k_bad },
    { KEY_NEXT,	"Next",	k_next,		k_bad,		k_bad,		k_bad },
};
#define N_KEYS (sizeof(key_fns) / sizeof(key_fns[0]))

/* External interface */

void
do_key(key_t key)
{
    int i;

    for (i=0; i < N_KEYS; i++) {
	if (key_fns[i].key != i) {
	    fprintf(stderr, "Key function table is skewed at element %d\n", i);
	    return;
	}
	if (key_fns[i].key == key) {
	    if (!Shift && !Ctrl){ (*key_fns[i].plain)(key);	return; }
	    if (Shift && !Ctrl)	{ (*key_fns[i].shift)(key);	return; }
	    if (!Shift && Ctrl)	{ (*key_fns[i].ctrl)(key);	return; }
	    if (Shift && Ctrl)	{ (*key_fns[i].shiftctrl)(key);	return; }
	}
    }
    fprintf(stderr, "Internal error: Impossible key value %d\n", key);
}
